/*
 * Pesaro_Audio_dwc
 * Driver for Pesaro Audio Codec dwc.
 *
 * Copyright (C) 2014  VATICS Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <linux/module.h>
#include <linux/io.h>
#include <linux/moduleparam.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/pm.h>
#include <linux/i2c.h>
#include <linux/slab.h>
#include <linux/regmap.h>
#include <linux/regulator/consumer.h>
#include <linux/spi/spi.h>
#include <linux/of_device.h>
#include <linux/clk.h>

#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/soc.h>
#include <sound/initval.h>
#include <sound/tlv.h>

#include "vatics-asoc.h"

/* dwc define */
#define ACDCC_VERSION                          0x00
#define ACDCC_CTRL                             0x04
#define ACDCC_RESET                            0x14
#define ACDCC_LATCH                            0x18
#define ACDCC_CLK_SEL0                         0x1C
#define ACDCC_CLK_SEL1                         0x20
#define ACDCC_I2S_CONFIG0                      0x24
#define ACDCC_I2S_CONFIG1                      0x28
#define ACDCC_I2S_CONFIG2                      0x2C
#define ACDCC_PWR_MANAGEMENT                   0x30 
#define ACDCC_PWR_CTRL0                        0x34
#define ACDCC_PWR_CTRL1                        0x38
#define ACDCC_PWR_CTRL3                        0x40
#define ACDCC_MUTE_CTRL0                       0x48 
#define ACDCC_MUTE_CTRL1                       0x4C
#define ACDCC_RECORD_VOL                       0x54
#define ACDCC_PGA_VOL                          0x5C
#define ACDCC_PLAYBACK_VOL                     0x64 
#define ACDCC_RECORD_INPUT_SEL                 0x6C 
#define ACDCC_PLAYBACK_MIXER_CTRL              0x78 
#define ACDCC_ALC_CTRL0                        0x80 
#define ACDCC_ALC_CTRL1                        0x84
#define ACDCC_ALC_CTRL2                        0x88
#define ACDCC_ALC_RMAP_DOWN_CTRL               0x8C
#define ACDCC_ALC_RMAP_UP_CTRL                 0x90 
#define ACDCC_ALC_MAX_AUTO_GAIN                0x94 
#define ACDCC_DIGITAL_NOISE_GATE               0x98 
#define ACDCC_DIGITAL_TEST                     0x9C 
#define ACDCC_ADC_HIGH_PASS_MIXER              0xA0 
#define ACDCC_DAC_MIXER                        0xA4 
#define ACDCC_SOFT_RAMPING                     0xA8 
#define ACDCC_DIGITAL_ASS_TEST1                0xB4 
#define ACDCC_DIGITAL_ASS_TEST2                0xB8 
#define ACDCC_DIGITAL_ASS_TEST3                0xBC
#define ACDCC_DIGITAL_ASS_TEST4                0xC0
#define ACDCC_DIGITAL_ASS_TEST5                0xC4 
#define ACDCC_PURE_ANALOG_TEST0                0xC8 
#define ACDCC_PURE_ANALOG_TEST1                0xCC 
#define ACDCC_PURE_ANALOG_TEST2                0xD0 
#define ACDCC_PURE_ANALOG_TEST3                0xD4 

#define VPL_SINGLE(xname, reg, shift, max, invert) \
{       .iface = SNDRV_CTL_ELEM_IFACE_MIXER, .name = xname, \
       .info = snd_soc_info_volsw, .get = dwc_audio_get_volsw,\
       .put = dwc_audio_put_volsw, \
       .private_value =  SOC_SINGLE_VALUE(reg, shift, max, invert) }

#define VPL_SINGLE_TLV(xname, reg, shift, max, invert, tlv_array) \
{       .iface = SNDRV_CTL_ELEM_IFACE_MIXER, .name = xname, \
       .access = SNDRV_CTL_ELEM_ACCESS_TLV_READ |\
       SNDRV_CTL_ELEM_ACCESS_READWRITE,\
       .tlv.p = (tlv_array), \
       .info = snd_soc_info_volsw, .get = dwc_audio_get_volsw,\
       .put = dwc_audio_put_volsw, \
       .private_value =  SOC_SINGLE_VALUE(reg, shift, max, invert) }


struct dwc_priv {
	unsigned int sysclk;
	enum snd_soc_control_type control_type;
};

static void *dwc_base;
static void *sysc_base;

inline void do_latch(void)
{
        mdelay(10);
	VATICS_ADCC_WRITEL(dwc_base + ACDCC_LATCH, 0x1);
        mdelay(10);
	VATICS_ADCC_WRITEL(dwc_base + ACDCC_LATCH, 0x0);

}

inline void dwc_data_path_reset(void)
{
	/* reset data path */
	VATICS_ADCC_WRITEL(dwc_base + ACDCC_RESET, 0x0); // set low
	do_latch();
	VATICS_ADCC_WRITEL(dwc_base + ACDCC_RESET, 0x3); // set high
	do_latch();
}

int dwc_audio_get_volsw(struct snd_kcontrol *kcontrol,
		struct snd_ctl_elem_value *ucontrol)
{
	struct soc_mixer_control *mc =
		(struct soc_mixer_control *)kcontrol->private_value;
	unsigned int reg = mc->reg;
	int max = mc->max;
	unsigned int invert = mc->invert;
	unsigned int val;

	val =  readl(IOMEM(dwc_base + reg));

	VPL_DEBUG("reg:0x%x val:0x%x\n", reg, val);

	if (invert)
		val = max - val;

	ucontrol->value.integer.value[0] = val;
	return 0;
}

int dwc_audio_put_volsw(struct snd_kcontrol *kcontrol,
		struct snd_ctl_elem_value *ucontrol)
{
	struct soc_mixer_control *mc =
		(struct soc_mixer_control *)kcontrol->private_value;
	unsigned int reg = mc->reg;
	int max = mc->max;
	unsigned int invert = mc->invert;
	unsigned int val;

	val = (ucontrol->value.integer.value[0] >= max) ?
				max : ucontrol->value.integer.value[0];
	VPL_DEBUG("reg:0x%x val:0x%x\n", reg, val);
	if(val == 0) {
		switch(reg) {
			case ACDCC_PLAYBACK_VOL:
				VATICS_ADCC_WRITEL(dwc_base + ACDCC_MUTE_CTRL1, 0x43);
				break;
			case ACDCC_RECORD_VOL:
				VATICS_ADCC_WRITEL(dwc_base + ACDCC_MUTE_CTRL0, 0x5);
				break;
			default:
				break;
		}
		VPL_DEBUG("MUTE!!!\n");
	} else {
		switch(reg) {
			case ACDCC_PLAYBACK_VOL:
				VATICS_ADCC_WRITEL(dwc_base + ACDCC_MUTE_CTRL1, 0x0);
				break;
			case ACDCC_RECORD_VOL:
				VATICS_ADCC_WRITEL(dwc_base + ACDCC_MUTE_CTRL0, 0x0);
				break;
			default:
				break;
		}

	}
	if (invert)
		val = max - val;

	VATICS_ADCC_WRITEL(dwc_base + reg, val);
	do_latch();
	return 1; // 1 means changed
}


static DECLARE_TLV_DB_SCALE(dwc_audio_spk_vol_tlv, -12600, 150, 1);
static DECLARE_TLV_DB_SCALE(dwc_audio_rec_vol_tlv, -9600, 150, 1);
static DECLARE_TLV_DB_SCALE(dwc_audio_pga_vol_tlv, -1800, 100, 0);

static const struct snd_kcontrol_new dwc_audio_snd_controls[] = {
	/* vatics standard interface */
	VPL_SINGLE_TLV("Playback Volume", ACDCC_PLAYBACK_VOL,
			0, 84, 0, dwc_audio_spk_vol_tlv),
	VPL_SINGLE_TLV("Capture Volume", ACDCC_RECORD_VOL,
			0, 84, 1, dwc_audio_rec_vol_tlv),
	VPL_SINGLE_TLV("PGA Volume", ACDCC_PGA_VOL,
			0, 52, 0, dwc_audio_pga_vol_tlv),
	/*0 = line-in, 2=mic-in(single-ended)*/
	VPL_SINGLE("Input Select", ACDCC_RECORD_INPUT_SEL,
			0, 7, 0),
	/* ALC controls */
	SOC_SINGLE("ALC_en", ACDCC_ALC_CTRL0,
			0, 3, 0),
	SOC_SINGLE("ALC_mxg", ACDCC_ALC_MAX_AUTO_GAIN,
			0, 255, 0),
	SOC_SINGLE("ALC_pk", ACDCC_ALC_CTRL1,
			0, 1, 0),
	SOC_SINGLE("ALC_trg", ACDCC_ALC_CTRL1,
			1, 3, 0),
	SOC_SINGLE("ALC_hld", ACDCC_ALC_CTRL2,
			0, 15, 0),
	SOC_SINGLE("ALC_rdn", ACDCC_ALC_RMAP_DOWN_CTRL,
			0, 65, 0),
	SOC_SINGLE("ALC_rup", ACDCC_ALC_RMAP_UP_CTRL,
			0, 65, 0),

};

__maybe_unused static int dwc_write(struct snd_soc_codec *codec, unsigned int reg, unsigned int val)
{
	int ret = 0;
	//struct dwc_priv *dwc = snd_soc_codec_get_drvdata(codec);
	return ret ;
}

__maybe_unused int dwc_read(struct snd_soc_codec* codec, int reg)
{
	int data = 0;
	return data;
}

static int dwc_hw_params(struct snd_pcm_substream *substream,
			    struct snd_pcm_hw_params *params,
			    struct snd_soc_dai *dai)
{
//TODO implement it
/*NOTE: I2S mode or sampling freq is change, reset to the data paths
 *      is recommended
 */      
	//struct snd_soc_codec *codec = dai->codec;
	VPL_DEBUG("%s +++\n", __func__);
	return 0;
}
static int dwc_mute(struct snd_soc_dai *dai, int mute)
{
	VPL_DEBUG("%s +++, mute:%d\n", __func__, mute);
	/*FIXME, the mute method may cause some noise, we do nothing in mute function now*/
	return 0;
	if (mute) {
		VATICS_ADCC_WRITEL(dwc_base + ACDCC_MUTE_CTRL1, 0x43);
	} else {
		VATICS_ADCC_WRITEL(dwc_base + ACDCC_MUTE_CTRL1, 0x0);
	}
	do_latch();
	//dwc_data_path_reset();
	return 0;
}

static int dwc_set_dai_clkdiv(struct snd_soc_dai *codec_dai,
		int div_id, int div)
{
	u32 reg=0;
	VPL_DEBUG("%s +++\n", __func__);
	switch (div) {
		case 96000:
			reg = 0xa;
			break;
		case 48000:
			reg = 0x8;
			break;
		case 44100:
			reg = 0x7;
			break;
		case 32000:
			reg = 0x6;
			break;
		case 22050:
			reg = 0x4;
			break;
		case 16000:
			reg = 0x3;
			break;
		case 11025:
			reg = 0x1;
			break;
		case 8000:
			reg = 0x0;
			break;
	}
	VATICS_ADCC_WRITEL(dwc_base + ACDCC_I2S_CONFIG1, reg);  // sfsdac
	VATICS_ADCC_WRITEL(dwc_base + ACDCC_I2S_CONFIG0, reg);  // sfsadc
	dwc_data_path_reset();
	return 0;
}

static int dwc_set_dai_fmt(struct snd_soc_dai *codec_dai,
		unsigned int fmt)
{
//TODO implement it
	VPL_DEBUG("%s +++\n", __func__);
	return 0;
}

static int dwc_trigger(struct snd_pcm_substream *substream, int cmd,
		struct snd_soc_dai *codec_dai)
{
	VPL_DEBUG("%s +++\n", __func__);
	if (substream->stream != SNDRV_PCM_STREAM_PLAYBACK)
		return 0;
	switch (cmd) {
		case SNDRV_PCM_TRIGGER_START:
			VATICS_ADCC_WRITEL(dwc_base + ACDCC_MUTE_CTRL1, 0x0);
			break;
		case SNDRV_PCM_TRIGGER_STOP:
			VATICS_ADCC_WRITEL(dwc_base + ACDCC_MUTE_CTRL1, 0x43);
			break;
	}
	do_latch();
	return 0;

}

static int dwc_set_bias_level(struct snd_soc_codec *codec,
				 enum snd_soc_bias_level level)
{
//TODO implement it
	switch (level) {
		case SND_SOC_BIAS_ON:
			//VPL_DEBUG("%s, SND_SOC_BIAS_ON\n", __func__);
			break;
		case SND_SOC_BIAS_PREPARE:
			//VPL_DEBUG("%s, SND_SOC_BIAS_PREPARE\n", __func__);
			break;
		case SND_SOC_BIAS_STANDBY:
			//VPL_DEBUG("%s, SND_SOC_BIAS_STANDBY\n", __func__);
			break;
		case SND_SOC_BIAS_OFF:
			//VPL_DEBUG("%s, SND_SOC_BIAS_OFF\n", __func__);
			break;
	}

	return 0;
}
static const struct snd_soc_dai_ops dwc_dai_ops = {
	.hw_params	= dwc_hw_params,
	.digital_mute	= dwc_mute,
	.set_clkdiv	= dwc_set_dai_clkdiv,
	.set_fmt	= dwc_set_dai_fmt,
	.trigger        = dwc_trigger,
};

static struct snd_soc_dai_driver dwc_dai = {
	.name = "dwc-hifi",
	.playback = {
		.stream_name = "dwc Playback",
		.channels_min = 1,
		.channels_max = 2,
		.rates = (SNDRV_PCM_RATE_8000 | SNDRV_PCM_RATE_16000 |\
			  SNDRV_PCM_RATE_11025 |SNDRV_PCM_RATE_22050 |\
			  SNDRV_PCM_RATE_32000 |SNDRV_PCM_RATE_44100 |\
			  SNDRV_PCM_RATE_96000 |\
			  SNDRV_PCM_RATE_48000 |SNDRV_PCM_RATE_KNOT),
		.formats = SNDRV_PCM_FMTBIT_S16_LE,
		},
	.capture = {
		.stream_name = "dwc Capture",
		.channels_min = 1,
		.channels_max = 2,
		.rates = (SNDRV_PCM_RATE_8000 | SNDRV_PCM_RATE_16000 |\
			  SNDRV_PCM_RATE_11025 |SNDRV_PCM_RATE_22050 |\
			  SNDRV_PCM_RATE_32000 |SNDRV_PCM_RATE_44100 |\
			  SNDRV_PCM_RATE_96000 |\
			  SNDRV_PCM_RATE_48000 |SNDRV_PCM_RATE_KNOT),
		.formats = SNDRV_PCM_FMTBIT_S16_LE,
		},
	.ops = &dwc_dai_ops,
	.symmetric_rates = 1,
};

__maybe_unused static void dump_reg(void)
{
	u32 reg;
	u32 offset = 0;
	int counter = 0;

	printk("\n");
	for (offset = 0 ; offset <= 0xd0 ; offset= offset+0x4, counter++ )
	{
		reg =  readl(IOMEM(dwc_base + offset));
		printk("0x%02x=0x%x  ", offset, reg);
		if ((counter % 4) == 3)
			printk("\n");
	}
	printk("\n");
	return;
}

unsigned int dwc_codec_read(struct snd_soc_codec *codec, unsigned int reg)
{
	return readl(dwc_base + reg);
}

int dwc_codec_write(struct snd_soc_codec *codec, unsigned int reg, unsigned int val)
{
	VATICS_ADCC_WRITEL(dwc_base + reg, val);
	return 0;
}

static int dwc_codec_probe(struct snd_soc_codec *codec)
{
	__maybe_unused struct dwc_priv *dwc = snd_soc_codec_get_drvdata(codec);
//	struct dwc_setup_data *pdata = codec->dev->platform_data;
	int ret = 0;
	struct clk *clk;
	VPL_DEBUG("%s +++\n", __func__);

/*NOTE: This is supposed to be standard io function settiong*/
#if 0
	ret = snd_soc_codec_set_cache_io(codec, 8, 16, dwc->control_type);
	if (ret < 0) {
		dev_err(codec->dev, "Failed to set cache I/O: %d\n", ret);
		return ret;
	}
#endif
	codec->read = dwc_codec_read;
	codec->write = dwc_codec_write;
	/* start the master clocks, mclkadc, mclkdac */
	clk = clk_get(NULL, "vpl_acdcc");
	if (IS_ERR(clk)) {
		dev_err(codec->dev, "Failed to get clock\n");
		return PTR_ERR(clk);
	}
	clk_prepare_enable(clk);

	// master reset
	VATICS_ADCC_WRITEL(dwc_base + ACDCC_CTRL, 0x0);
	// need delay here?
	VATICS_ADCC_WRITEL(dwc_base + ACDCC_CTRL, 0x1);

	// soft reset
	VATICS_ADCC_WRITEL(dwc_base + ACDCC_CTRL, 0x3);
	// need delay here?
	
	// pdz0
	VATICS_ADCC_WRITEL(dwc_base + ACDCC_PWR_CTRL0, 0x0); //R21
	do_latch();

	// R2 setting mclkadc 12M
	VATICS_ADCC_WRITEL(dwc_base + ACDCC_CLK_SEL0, 0x2);
        mdelay(10);
	// R3 setting mclkdac 12M
	VATICS_ADCC_WRITEL(dwc_base + ACDCC_CLK_SEL1, 0x2);
	do_latch();

	// R21 Mic Bias 1 and vcmbuf power on
	VATICS_ADCC_WRITEL(dwc_base + ACDCC_PWR_CTRL0, 0x6);
	// R22 PGA and ADC power on
	VATICS_ADCC_WRITEL(dwc_base + ACDCC_PWR_CTRL1, 0x5);
	// R24 PGA and ADC power on
	VATICS_ADCC_WRITEL(dwc_base + ACDCC_PWR_CTRL3, 0x89);
	do_latch();


	VATICS_ADCC_WRITEL(dwc_base + ACDCC_I2S_CONFIG2, 0x19);// set master mode
	do_latch();
	dwc_data_path_reset();

	/* set pdz to high */
	VATICS_ADCC_WRITEL(dwc_base + ACDCC_PWR_CTRL0, 0x5); // power-up all and MicBias1
	do_latch();

	/*set digital playback source*/
	VATICS_ADCC_WRITEL(dwc_base + ACDCC_PLAYBACK_MIXER_CTRL, 0x0);

	/*set input line1p*/
	VATICS_ADCC_WRITEL(dwc_base + ACDCC_RECORD_INPUT_SEL, 0x0);
	/*mic-input*/
	//VATICS_ADCC_WRITEL(dwc_base + ACDCC_RECORD_INPUT_SEL, 0x2);
	do_latch();

	// otus modify, enable hi-pass filter
	VATICS_ADCC_WRITEL(dwc_base + ACDCC_ADC_HIGH_PASS_MIXER, 0x04);
	do_latch();

	dwc_set_bias_level(codec, SND_SOC_BIAS_STANDBY);

	ret = snd_soc_add_codec_controls(codec, dwc_audio_snd_controls,
			     ARRAY_SIZE(dwc_audio_snd_controls));
	if (ret)
		return ret;
//	dump_reg();
	return ret;
}

static int dwc_codec_remove(struct snd_soc_codec *codec)
{
	VPL_DEBUG("%s +++\n", __func__);
	dwc_set_bias_level(codec, SND_SOC_BIAS_OFF);
	return 0;
}

#ifdef CONFIG_PM
static int dwc_codec_suspend(struct snd_soc_codec *codec)
{
	VPL_DEBUG("%s +++\n", __func__);
	dwc_set_bias_level(codec, SND_SOC_BIAS_OFF);
	return 0;
}

static int dwc_codec_resume(struct snd_soc_codec *codec)
{
	VPL_DEBUG("%s +++\n", __func__);
	dwc_set_bias_level(codec, SND_SOC_BIAS_STANDBY);
	return 0;
}
#else
#define dwc_codec_suspend NULL
#define dwc_codec_resume NULL
#endif

static struct snd_soc_codec_driver soc_codec_dev_dwc = {
	.probe =	dwc_codec_probe,
	.remove =	dwc_codec_remove,
	.suspend =	dwc_codec_suspend,
	.resume =	dwc_codec_resume,
	.set_bias_level = dwc_set_bias_level,
/*TODO add widget/control functions*/
#if 0
	.dapm_widgets = dwc_dapm_widgets,
	.num_dapm_widgets = ARRAY_SIZE(dwc_dapm_widgets),
	.dapm_routes = dwc_intercon,
	.num_dapm_routes = ARRAY_SIZE(dwc_intercon),
	.controls =	dwc_snd_controls,
	.num_controls = ARRAY_SIZE(dwc_snd_controls),
#endif
};

#if 0
static const struct of_device_id dwc_audio_of_match[] = {
	{ .compatible = "snps,designware-audio", },
	{},
};
MODULE_DEVICE_TABLE(of, dwc_audio_of_match);
#endif

static int dwc_driver_probe(struct platform_device *pdev)
{
	int ret;
	struct dwc_priv *dwc;
	VPL_DEBUG("%s +++\n", __func__);

	dwc = kzalloc(sizeof(struct dwc_priv), GFP_KERNEL);
	if (dwc == NULL)
		return -ENOMEM;
	request_mem_region(VPL_ACDCC_MMR_BASE, 0x100, "ACDCC");
	dwc_base = ioremap(VPL_ACDCC_MMR_BASE,0x100);

	request_mem_region(VPL_SYSC_MMR_BASE, 0x100, "SYSC");
	sysc_base = ioremap(VPL_SYSC_MMR_BASE,0x100);

	ret = snd_soc_register_codec(&pdev->dev,
			&soc_codec_dev_dwc, &dwc_dai, 1);
	if(ret < 0) {
		printk("dwc failed to init!!\n");
		goto err;
	}
	VPL_DEBUG("%s ---\n", __func__);
	return ret;

err:
	return ret;


}

static int dwc_driver_exit(struct platform_device *pdev)
{
	iounmap(dwc_base);
	iounmap(sysc_base);
	snd_soc_unregister_codec(&pdev->dev);
	return 0;
}

MODULE_ALIAS("platform:audio_designware");

struct platform_driver dwc_audio_driver = {
	.probe          = dwc_driver_probe,
	.remove         = dwc_driver_exit,
	.driver         = {
		.name   = "dwc_audio",
		.owner  = THIS_MODULE,
//		.of_match_table = of_match_ptr(dwc_audio_of_match),
	},
};


EXPORT_SYMBOL(dwc_audio_driver);

MODULE_DESCRIPTION("ASoC dw audio driver");
MODULE_LICENSE("GPL");
