/*
 * Audio_SSM2603
 * Driver for Audio Codec SSM2603.
 *
 * Copyright (C) 2010  VATICS Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/pm.h>
#include <linux/i2c.h>
#include <linux/slab.h>
#include <linux/regmap.h>
#include <linux/regulator/consumer.h>
#include <linux/spi/spi.h>
#include <linux/of_device.h>

#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/soc.h>
#include <sound/initval.h>
#include <sound/tlv.h>

#include "vatics-asoc.h"

/* SSM2603 define */
#define LEFT_ADC_VOLUME_REG             0x00
#define RIGHT_ADC_VOLUME_REG    0x01
#define LEFT_DAC_VOLUME_REG             0x02
#define RIGHT_DAC_VOLUME_REG    0x03
#define ANA_AUDIO_PATH_REG              0x04
#define DIG_AUDIO_PATH_REG              0x05
#define POWER_MANAGE_REG                0x06
#define DIGI_AUDIO_IF_REG               0x07
#define SAMPLE_RATE_REG                 0x08
#define ACTIVE_REG                              0x09
#define SW_RESET_REG                    0x0F
#define ALC_CTRL1_REG                   0x10
#define ALC_CTRL2_REG                   0x11
#define NOISE_GATE_REG                  0x12
static int ssm2603_reg_cache[18] ={ 0 } ;

struct ssm2603_priv {
	unsigned int sysclk;
	enum snd_soc_control_type control_type;
	struct i2c_client *i2c_client;
};

static int ssm2603_i2c_write(struct snd_soc_codec *codec, unsigned int reg, unsigned int val)
{
	struct ssm2603_priv *ssm2603 = snd_soc_codec_get_drvdata(codec);
	struct i2c_client *i2cdev = ssm2603->i2c_client;
	unsigned char buf[2] ;
	unsigned char recv[2] = {0} ;
	int ret = 0 ;
	buf[0] = (unsigned char)((reg << 1) | ((val >> 8) & 0x01));
	buf[1] = (unsigned char)(val & 0xff) ;

	if(i2c_master_send(i2cdev, buf, 2) != 2) {
		ret = -1 ;
	}
	ssm2603_reg_cache[reg] = val ;

	recv[0] = (unsigned char)(reg << 1) ;
	recv[1] = 0 ;
	i2c_master_recv(i2cdev, recv, 2) ;
	return ret ;
}

__maybe_unused int vpl_audio_read_i2cdev_reg(struct i2c_client *i2cdev, int reg)
{
	int data = ssm2603_reg_cache[reg] ;

	return data;
}


#define ssm2603_reset(c)	snd_soc_write(c, SW_RESET_REG, 0)

int codec_init(struct snd_soc_codec *codec)
{
	int ret = 0 ;
	VPL_DEBUG("%s +++\n", __func__);

	//init every registers..
	ret = 0 ;
	ret |= snd_soc_write(codec, LEFT_ADC_VOLUME_REG, 0x17);	// LLINE_VOLCTRL
	ret |= snd_soc_write(codec, RIGHT_ADC_VOLUME_REG, 0x17);	// RLINE_VOLCTRL
	ret |= snd_soc_write(codec, LEFT_DAC_VOLUME_REG, 0x79);	// LHEADPHONE_VOLCTRL
	ret |= snd_soc_write(codec, RIGHT_DAC_VOLUME_REG, 0x79);	// RHEADPHONE_VOLCTRL
	ret |= snd_soc_write(codec, ANA_AUDIO_PATH_REG, 0x113);	// ANA_AUDIOPATH_CTRL
	ret |= snd_soc_write(codec, DIG_AUDIO_PATH_REG, 0x00);	// DIGI_AUDIOPATH_CTRL
	ret |= snd_soc_write(codec, POWER_MANAGE_REG, 0x60);		// POWERDOWN_CTRL
	ret |= snd_soc_write(codec, DIGI_AUDIO_IF_REG, 0x42);		// DIGI_INTERFACE_FORMAT
	ret |= snd_soc_write(codec, SAMPLE_RATE_REG, 0x00) ;		// SAMPLERATE_CTRL
	ret |= snd_soc_write(codec, ACTIVE_REG, 0x01) ;			// ACTIVE

	if(ret != 0) {
		printk("[ERR] Initialize SSM2603 failed!!\n") ;
		return -1 ;
	}
	return 0 ;
}

static int ssm2603_hw_params(struct snd_pcm_substream *substream,
			    struct snd_pcm_hw_params *params,
			    struct snd_soc_dai *dai)
{
//TODO implement it
	//struct snd_soc_codec *codec = dai->codec;
	VPL_DEBUG("%s +++\n", __func__);
	return 0;
}
static int ssm2603_mute(struct snd_soc_dai *dai, int mute)
{
//TODO implement it
	VPL_DEBUG("%s +++\n", __func__);
	return 0;
}

static int ssm2603_set_dai_sysclk(struct snd_soc_dai *codec_dai,
		int clk_id, unsigned int freq, int dir)
{
//TODO implement it
	VPL_DEBUG("%s +++\n", __func__);
	return 0;
}

static int ssm2603_set_dai_fmt(struct snd_soc_dai *codec_dai,
		unsigned int fmt)
{
//TODO implement it
	VPL_DEBUG("%s +++\n", __func__);
	return 0;
}

static int ssm2603_set_bias_level(struct snd_soc_codec *codec,
				 enum snd_soc_bias_level level)
{
//TODO implement it
	switch (level) {
		case SND_SOC_BIAS_ON:
			VPL_DEBUG("%s, SND_SOC_BIAS_ON\n", __func__);
			break;
		case SND_SOC_BIAS_PREPARE:
			VPL_DEBUG("%s, SND_SOC_BIAS_PREPARE\n", __func__);
			break;
		case SND_SOC_BIAS_STANDBY:
			VPL_DEBUG("%s, SND_SOC_BIAS_STANDBY\n", __func__);
			break;
		case SND_SOC_BIAS_OFF:
			VPL_DEBUG("%s, SND_SOC_BIAS_OFF\n", __func__);
			break;
	}

	return 0;
}
static const struct snd_soc_dai_ops ssm2603_dai_ops = {
	.hw_params	= ssm2603_hw_params,
	.digital_mute	= ssm2603_mute,
	.set_sysclk	= ssm2603_set_dai_sysclk,
	.set_fmt	= ssm2603_set_dai_fmt,
};

static struct snd_soc_dai_driver ssm2603_dai = {
	.name = "ssm2603-hifi",
	.playback = {
		.stream_name = "Playback",
		.channels_min = 1,
		.channels_max = 2,
		.rates = (SNDRV_PCM_RATE_8000 | SNDRV_PCM_RATE_16000 |\
			  SNDRV_PCM_RATE_32000 |SNDRV_PCM_RATE_44100 |\
			  SNDRV_PCM_RATE_48000 |SNDRV_PCM_RATE_KNOT),
		.formats = SNDRV_PCM_FMTBIT_S16_LE,
		},
	.capture = {
		.stream_name = "Capture",
		.channels_min = 1,
		.channels_max = 2,
		.rates = (SNDRV_PCM_RATE_8000 | SNDRV_PCM_RATE_16000 |\
			  SNDRV_PCM_RATE_32000 |SNDRV_PCM_RATE_44100 |\
			  SNDRV_PCM_RATE_48000 |SNDRV_PCM_RATE_KNOT),
		.formats = SNDRV_PCM_FMTBIT_S16_LE,
		},
	.ops = &ssm2603_dai_ops,
	.symmetric_rates = 1,
};

static int ssm2603_probe(struct snd_soc_codec *codec)
{
	__maybe_unused struct ssm2603_priv *ssm2603 = snd_soc_codec_get_drvdata(codec);
//	struct ssm2603_setup_data *pdata = codec->dev->platform_data;
	int ret;
	VPL_DEBUG("%s +++\n", __func__);

/*NOTE: This is supposed to be standard io function settiong*/
#if 0
	ret = snd_soc_codec_set_cache_io(codec, 8, 16, ssm2603->control_type);
	if (ret < 0) {
		dev_err(codec->dev, "Failed to set cache I/O: %d\n", ret);
		return ret;
	}
#endif

	ret = ssm2603_reset(codec);
	if (ret < 0) {
		dev_err(codec->dev, "Failed to issue reset\n");
		return ret;
	}

	ret = codec_init(codec);
	if (ret < 0) {
		return ret;
	}

	ssm2603_set_bias_level(codec, SND_SOC_BIAS_STANDBY);

//TODO add widgets
#if 0
	ret = snd_soc_add_codec_controls(codec, ssm2603_snd_controls,
			     array_size(ssm2603_snd_controls));
	if (ret)
		return ret;
	ret = ssm2603_add_widgets(codec);
#endif
	return ret;
}

static int ssm2603_remove(struct snd_soc_codec *codec)
{
	VPL_DEBUG("%s +++\n", __func__);
	ssm2603_set_bias_level(codec, SND_SOC_BIAS_OFF);
	return 0;
}

#ifdef CONFIG_PM
static int ssm2603_suspend(struct snd_soc_codec *codec)
{
	VPL_DEBUG("%s +++\n", __func__);
	ssm2603_set_bias_level(codec, SND_SOC_BIAS_OFF);
	return 0;
}

static int ssm2603_resume(struct snd_soc_codec *codec)
{
	VPL_DEBUG("%s +++\n", __func__);
	ssm2603_set_bias_level(codec, SND_SOC_BIAS_STANDBY);
	return 0;
}
#else
#define ssm2603_suspend NULL
#define ssm2603_resume NULL
#endif

static struct snd_soc_codec_driver soc_codec_dev_ssm2603 = {
	.probe =	ssm2603_probe,
	.remove =	ssm2603_remove,
	.suspend =	ssm2603_suspend,
	.resume =	ssm2603_resume,
	.set_bias_level = ssm2603_set_bias_level,
	.write          = ssm2603_i2c_write,
/*TODO add widget/control functions*/
#if 0
	.dapm_widgets = ssm2603_dapm_widgets,
	.num_dapm_widgets = ARRAY_SIZE(ssm2603_dapm_widgets),
	.dapm_routes = ssm2603_intercon,
	.num_dapm_routes = ARRAY_SIZE(ssm2603_intercon),
	.controls =	ssm2603_snd_controls,
	.num_controls = ARRAY_SIZE(ssm2603_snd_controls),
#endif
};

static const struct of_device_id ssm2603_of_match[] = {
	{ .compatible = "wlf,ssm2603", },
	{ }
};
MODULE_DEVICE_TABLE(of, ssm2603_of_match);

static int ssm2603_i2c_probe(struct i2c_client *client,
			    const struct i2c_device_id *id)
{
	int ret;
	struct ssm2603_priv *ssm2603;
	VPL_DEBUG("%s +++\n", __func__);

	ssm2603 = kzalloc(sizeof(struct ssm2603_priv), GFP_KERNEL);

	if (ssm2603 == NULL)
		return -ENOMEM;
//TODO: add support for i2sClkSrc
#if 0
	/* Setup PLL clock source */
	if(i2sClkSrc) {
		reg =  readl(IOMEM((DWORD)sysc_base + SYSC_IF_CTRL));
		reg |=  (0x1<<25); /* PLL2 Source 0:18.432 MHz / 1:24 MHz*/
		writel(reg, IOMEM((DWORD)sysc_base + SYSC_IF_CTRL));
	}
#endif

	i2c_set_clientdata(client, ssm2603);
	ssm2603->control_type = SND_SOC_I2C;
	ssm2603->i2c_client = client;

	ret = snd_soc_register_codec(&client->dev,
			&soc_codec_dev_ssm2603, &ssm2603_dai, 1);
	if(ret < 0) {
		printk("SSM2603 failed to init!!\n");
		goto err;
	}
	VPL_DEBUG("%s ---\n", __func__);
	return ret;

err:
	return ret;


}

static int ssm2603_i2c_remove(struct i2c_client *client)
{
	snd_soc_unregister_codec(&client->dev);
	return 0;
}

static const struct i2c_device_id ssm2603_i2c_id[] = {
	{ "ssm2603", 0 },
	{ }
};
MODULE_DEVICE_TABLE(i2c, ssm2603_i2c_id);

static const unsigned short normal_i2c[] = { 0x1a, I2C_CLIENT_END };
struct i2c_driver ssm2603_i2c_driver = {
	.driver = {
		.name = "ssm2603",
		.owner = THIS_MODULE,
		.of_match_table = ssm2603_of_match,
	},
	.probe =    ssm2603_i2c_probe,
	.remove =   ssm2603_i2c_remove,
	.id_table = ssm2603_i2c_id,
	.address_list = normal_i2c,
	.class = I2C_CLASS_HWMON,
};
//module_i2c_driver(ssm2603_i2c_driver);
EXPORT_SYMBOL(ssm2603_i2c_driver);

MODULE_DESCRIPTION("ASoC ssm2603 driver");
MODULE_LICENSE("GPL");
