/*
 *
 * Copyright (C) 2014  VATICS Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/device.h>
#include <linux/delay.h>
#include <linux/io.h>
#include <linux/clk.h>

#include <linux/of.h>

#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/initval.h>
#include <sound/soc.h>

#include "vatics-asoc.h"


#define VAUDIO_IIS_DRIVER_VERSION		"5.0.0.1"
#define VAUDIO_IIS_COMPANY			"VATICS Inc."
#define VAUDIO_IIS_PLATFORM			"EVM"

const char vaudioiis_id[] = "$VaudioIis: "VAUDIO_IIS_DRIVER_VERSION" "VAUDIO_IIS_PLATFORM" "__DATE__" "VAUDIO_IIS_COMPANY" $";


// SYSC
#define SYSC_BASE               		VPL_SYSC_MMR_BASE
#define SYSC_ID_VERSION       			0x00000000
#define SYSC_I2S_CLK_PARAM 			0x0000002C
#define SYSC_I2S_CLK_ENABLE			0x00000028
#define SYSC_I2S_ENABLE				0x00000800
#define SYSC_IF_CTRL                    	0x00000048

// register offset
#define SYSC_PAD_EN_CTRL_5                      0x000000B0
#define SYSC_I2SSC_CTRL_0                       0x00000058
#define SYSC_I2SSC_CTRL_1                       0x0000005C

// SYSC_I2SSC_CTRL_0, shift
#define I2SSC_TX_WS_DIV                         24
#define I2SSC_TX_SCLK_DIV                       16
#define I2SSC_TX_MCLK_DIV                       8
#define I2SSC_RX_MCLK_DIV                       0
// SYSC_I2SSC_CTRL_1, shift
#define I2SSC_MCLK_PAD_OUT_DRV_CTRL             20
#define I2SSC_PAD_OUT_DRV_CTRL                  16
#define I2SSC_MCLK_PAD_SR_CTRL                  12
#define I2SSC_TX_DST_SRC_SEL                    8
#define I2SSC_TX_SCLK_SRC_SEL                   4
#define I2SSC_RX_SCLK_SRC_SEL                   0

#define PLLC_5_CTRL                             0x2C
#define PLLC_5_DIV                              0x30

static void *g_sysc_base;
void *pllc_base;

void set_pll5(int target)
{
	/* default, target = pll5 / mclk_div,
	 *                 = pll5 / 8
	 */
	u32 reg;
	u32 current_reg;
	switch (target) {
		// use one pll table here
		case 12000000:
			reg = 0x7cf0918;
			break;
		case 49152000:
			reg = 0x7ff1318;
			break;
		case 24576000:
			reg = 0x7ff1331;
			break;
		default:
			printk("unknown target, do nothing\n");
			return;
	}
	current_reg = readl(IOMEM((unsigned long)pllc_base + PLLC_5_DIV));
	if (current_reg == reg)
		return;

	writel(reg, IOMEM((unsigned long)pllc_base + PLLC_5_DIV));
	reg = readl(IOMEM((unsigned long)pllc_base + PLLC_5_CTRL));
	reg |= 0x1;
	writel(reg, IOMEM((unsigned long)pllc_base + PLLC_5_CTRL));
	while (readl(IOMEM((unsigned long)pllc_base + PLLC_5_CTRL)) != 0x30) {
		printk("PLL Unlock....\n");
	}

}

static void __i2s_init(struct snd_pcm_substream *substream, int direct)
{
	/* Initialize I2S */
	unsigned int dev_tx = ssp_tx[0];
	unsigned int dev_rx = ssp_rx[0];
	unsigned int virt_dev_tx = __IO_ADDRESS(dev_tx);
	unsigned int virt_dev_rx = __IO_ADDRESS(dev_rx);
	u32 reg = 0;
	static int isInit = 1;
	__maybe_unused struct snd_pcm *pcm = substream->pcm;
	struct clk *clk;

	/* init the mclk output for codec*/
	if (isInit) {

		reg = readl(IOMEM((unsigned long)g_sysc_base + SYSC_PAD_EN_CTRL_5));
//#define MASTER
#ifdef CONFIG_SND_PESARO_EXTERNAL_CODEC
		reg|= 1<<10;    //I2SSC_TXPAD_PAD_EN
		reg|= 1<<9;     //I2SSC_RXPAD_PAD_EN
		reg&= ~(1<<20); //N903U PAD EN, turn off (share pin with i2ss external)
#ifdef MASTER
		reg|= 1<<12;  //I2SSC_MASTER
#endif
		reg|= 1<<11;  //I2SSC_MCLK_PAD_EN
#endif
		writel(reg, IOMEM((unsigned long)g_sysc_base + SYSC_PAD_EN_CTRL_5));

		clk = clk_get(NULL, "vpl_i2ssc");
		if (IS_ERR(clk)) {
			printk(KERN_ERR "[I2S] Failed to get clock\n");
			return;
		}
		clk_prepare_enable(clk);

		isInit = 0;
	}

	if (direct == SNDRV_PCM_STREAM_PLAYBACK) {
		VATICS_VPL_SSP_WRITEL(virt_dev_tx, I2S_IER, 0x0);
		VATICS_VPL_SSP_WRITEL(virt_dev_tx, I2S_IRER, 0x0);
		VATICS_VPL_SSP_WRITEL(virt_dev_tx, I2S_ITER, 0x0);
		VATICS_VPL_SSP_WRITEL(virt_dev_tx, I2S_IMR, 0x0);
		VATICS_VPL_SSP_WRITEL(virt_dev_tx, I2S_RFCR, 0x3);
		VATICS_VPL_SSP_WRITEL(virt_dev_tx, I2S_TFCR, 0x3);
		VATICS_VPL_SSP_WRITEL(virt_dev_tx, I2S_RFF0, 0x1);
		VATICS_VPL_SSP_WRITEL(virt_dev_tx, I2S_TFF0, 0x1);
		VATICS_VPL_SSP_WRITEL(virt_dev_tx, I2S_RCR, 0x2);
		VATICS_VPL_SSP_WRITEL(virt_dev_tx, I2S_TCR, 0x2);
	} else {
		VATICS_VPL_SSP_WRITEL(virt_dev_rx, I2S_IER, 0x0);
		VATICS_VPL_SSP_WRITEL(virt_dev_rx, I2S_IRER, 0x0);
		VATICS_VPL_SSP_WRITEL(virt_dev_rx, I2S_ITER, 0x0);
		VATICS_VPL_SSP_WRITEL(virt_dev_rx, I2S_IMR, 0x0);
		VATICS_VPL_SSP_WRITEL(virt_dev_rx, I2S_RFCR, 0x3);
		VATICS_VPL_SSP_WRITEL(virt_dev_rx, I2S_TFCR, 0x3);
		VATICS_VPL_SSP_WRITEL(virt_dev_rx, I2S_RFF0, 0x1);
		VATICS_VPL_SSP_WRITEL(virt_dev_rx, I2S_TFF0, 0x1);
		VATICS_VPL_SSP_WRITEL(virt_dev_rx, I2S_RCR, 0x2);
		VATICS_VPL_SSP_WRITEL(virt_dev_rx, I2S_TCR, 0x2);
	}
}

static int vatics_i2s_startup(struct snd_pcm_substream *substream, struct snd_soc_dai *dai)
{
	VPL_DEBUG("%s +++\n", __func__);
	__i2s_init(substream, substream->stream);
		return 0;
}

static int vatics_i2s_set_dai_fmt(struct snd_soc_dai *cpu_dai, unsigned int fmt)
{
//	struct vatics_audio_dev *dev = snd_soc_dai_get_drvdata(cpu_dai);
	VPL_DEBUG("%s +++\n", __func__);
	/*DO NOTHING for current design, leave this for template*/
	return 0;
}
static int vatics_i2s_set_dai_clkdiv(struct snd_soc_dai *cpu_dai, int div_id, int div)
{
	u32 reg = 0;
	VPL_DEBUG("%s +++\n", __func__);
	if (div_id == 0) {// internal or external codec
		/* set Tx/Rx clock source, data destination*/
#if defined(CONFIG_SND_PESARO_EXTERNAL_CODEC) || defined (CONFIG_SND_PESARO_EXTERNAL_CODEC_MODULE)
		reg = 0x0;
		reg|= 0x1<<I2SSC_TX_DST_SRC_SEL; // TX data DST, external
#ifdef MASTER
		reg|= 0x2<<I2SSC_TX_SCLK_SRC_SEL; // TX sclk source, from PLL
#else
		reg|= 0x1<<I2SSC_TX_SCLK_SRC_SEL; // TX sclk source, from external
#endif // end of MASTER
		reg|= 0x1; // RX sclk source, from external
#else
		// internal codec
		reg = 0x0;
#endif //end of EXTERNAL
	} else {
		switch (div) {
			case 48000:
				set_pll5(24576000);
				writel(0x1f000303, 
					IOMEM((unsigned long)g_sysc_base + SYSC_I2SSC_CTRL_0));  // set 64 fs, DIV=8, for 48k
				break;
/*XXX the 22579200 pll clock seems incorrect, remove it
 *    we dont use the clock for hdmi, only 48k/32k now
 */
#if 0
			case 44100:
				set_pll5(22579200);
				writel(0x1f000003, 
					IOMEM((unsigned long)g_sysc_base + SYSC_I2SSC_CTRL_0));  // set 64 fs,  for 441k
				break;
#endif
			case 32000:
				set_pll5(24576000);
				writel(0x1f000503, 
					IOMEM((unsigned long)g_sysc_base + SYSC_I2SSC_CTRL_0));  // set 64 fs, DIV=12, for 32k
				break;
			default:
				pr_warning("non-support sample rate:%d!\n", div);
				break;
		};
		reg|= 2<<I2SSC_TX_DST_SRC_SEL;
		reg|= 2<<I2SSC_TX_SCLK_SRC_SEL;
	}
	writel(reg, IOMEM((unsigned long)g_sysc_base + SYSC_I2SSC_CTRL_1));
	return 0;
}

static int vatics_i2s_set_dai_pll(struct snd_soc_dai *codec_dai,
				  int pll_id, int source,
				  unsigned int freq_in,
				  unsigned int freq_out)
{
	VPL_DEBUG("%s +++\n", __func__);
	/*DO NOTHING for current design, leave this for template*/
	set_pll5(freq_out);
	return 0;
}

static int vatics_i2s_set_sysclk(struct snd_soc_dai *dai, int clk_id,
				    unsigned int freq, int dir)
{
	//struct davinci_audio_dev *dev = snd_soc_dai_get_drvdata(dai);
	VPL_DEBUG("%s +++\n", __func__);
	/*DO NOTHING for current design, leave this for template*/
	return 0;
}

static int vatics_i2s_hw_params(struct snd_pcm_substream *substream, struct snd_pcm_hw_params *params, struct snd_soc_dai *cpu_dai)
{
//	struct davinci_audio_dev *dev = snd_soc_dai_get_drvdata(cpu_dai);
	VPL_DEBUG("%s +++\n", __func__);
	/*DO NOTHING for current design, leave this for template*/
	return 0;
}

static void vatics_i2s_enable_device(int dev_num, int enable, int stream_id)
{
	unsigned int dev_tx = ssp_tx[dev_num];
	unsigned int dev_rx = ssp_rx[dev_num];
	unsigned int virt_dev_tx = __IO_ADDRESS(dev_tx);
	unsigned int virt_dev_rx = __IO_ADDRESS(dev_rx);
	VPL_DEBUG("%s +++\n", __func__);
	if (stream_id == SNDRV_PCM_STREAM_PLAYBACK) {
		VATICS_VPL_SSP_WRITEL(virt_dev_tx, I2S_IER, (enable) ? 0x1 : 0x0);
	} else {
		VATICS_VPL_SSP_WRITEL(virt_dev_rx, I2S_IER, (enable) ? 0x1 : 0x0);
	}
}

 static void vatics_i2s_enable_stream(int dev_num, int enable, int stream_id)
{
	unsigned int dev_tx = ssp_tx[dev_num];
	unsigned int dev_rx = ssp_rx[dev_num];
	unsigned int virt_dev_tx = __IO_ADDRESS(dev_tx);
	unsigned int virt_dev_rx = __IO_ADDRESS(dev_rx);

	VPL_DEBUG("%s +++\n", __func__);
	if (stream_id == SNDRV_PCM_STREAM_PLAYBACK) {
		VATICS_VPL_SSP_WRITEL(virt_dev_tx, I2S_ITER, (enable) ? 0x1 : 0x0);
	}
	else {
		VATICS_VPL_SSP_WRITEL(virt_dev_rx, I2S_IRER, (enable) ? 0x1 : 0x0);
	}
}

static int vatics_i2s_trigger(struct snd_pcm_substream *substream, int cmd, struct snd_soc_dai *cpu_dai)
{
	struct vatics_runtime_data *prtd = substream->runtime->private_data;
	int ret = 0;

	VPL_DEBUG("%s +++\n", __func__);
	spin_lock(&prtd->lock);
	switch (cmd) {
	case SNDRV_PCM_TRIGGER_START:
		vatics_i2s_enable_device(prtd->dev_num, 1, substream->stream);
		vatics_i2s_enable_stream(prtd->dev_num, 1, substream->stream);
		break;

	case SNDRV_PCM_TRIGGER_STOP:
		vatics_i2s_enable_stream(prtd->dev_num, 0, substream->stream);
		break;
	default:
		ret = -EINVAL;
		break;
	}
	spin_unlock(&prtd->lock);
	return 0;
}

static const struct snd_soc_dai_ops vatics_i2s_dai_ops = {
	.startup = vatics_i2s_startup,
	.trigger = vatics_i2s_trigger,
	.hw_params = vatics_i2s_hw_params,
	.set_fmt = vatics_i2s_set_dai_fmt,
	.set_clkdiv = vatics_i2s_set_dai_clkdiv,
	.set_sysclk = vatics_i2s_set_sysclk,
	.set_pll = vatics_i2s_set_dai_pll,
};
#define VATICS_I2S_RATES        SNDRV_PCM_RATE_8000_48000 | SNDRV_PCM_RATE_96000
static struct snd_soc_dai_driver vatics_i2s_dai[] = {
	{
		.playback	= {
			.channels_min = 2,
			.channels_max = 2,
			.rates = VATICS_I2S_RATES,
			.formats = SNDRV_PCM_FMTBIT_S16_LE,
		},
		.capture 	= {
			.channels_min = 2,
			.channels_max = 2,
			.rates = VATICS_I2S_RATES,
			.formats = SNDRV_PCM_FMTBIT_S16_LE,
		},
		.ops 		= &vatics_i2s_dai_ops,
	},
};

static const struct snd_soc_component_driver vatics_i2s_component = {
	.name		= "vatics-i2s",
};

static const struct of_device_id vatics_i2s_dt_ids[] = {
	{
		.compatible = "vatics,vpl-audio-i2s",
	},
	{ /* sentinel */ }
};
MODULE_DEVICE_TABLE(of, vatics_i2s_dt_ids);

static int vatics_i2s_probe(struct platform_device *pdev)
{
	struct vatics_audio_dev *dev;
	int ret;
	VPL_DEBUG("%s +++\n", __func__);

	dev = devm_kzalloc(&pdev->dev, sizeof(struct vatics_audio_dev),
			   GFP_KERNEL);
/* TODO: set platform data here or read dts setting here */
#if 0
	const struct of_device_id *match =
			of_match_device(of_match_ptr(vatics_i2s_dt_ids), &pdev->dev);

	if (!pdev->dev.platform_data && !pdev->dev.of_node) {
		dev_err(&pdev->dev, "No platform data supplied\n");
		return -EINVAL;
	}

	if (!dev)
		return	-ENOMEM;
	if (pdev->dev.platform_data) {
		pdata = pdev->dev.platform_data;
	} else if (match) {
		pdata = devm_kzalloc(&pdev->dev, sizeof(*pdata), GFP_KERNEL);
		if (!pdata) {
			ret = -ENOMEM;
			goto nodata;
		}
	} else {
		ret = -EINVAL;
		goto nodata;
	}
	of_property_read_u32(np, "xxx", &val)
#endif
#if 0
	pm_runtime_enable(&pdev->dev);

	ret = pm_runtime_get_sync(&pdev->dev);
#endif
	request_mem_region(SYSC_BASE, 0x100, "SYSC");
	dev->sysc_base = ioremap(SYSC_BASE, 0x100);
	g_sysc_base = dev->sysc_base;

	pllc_base = ioremap(VPL_PLLC_MMR_BASE, 0x100);

	dev->dev = &pdev->dev;

	ret = snd_soc_register_component(&pdev->dev, &vatics_i2s_component,
					 &vatics_i2s_dai[0], 1);
	if (ret != 0)
		goto err_release_clk;

	ret = vatics_soc_platform_register(&pdev->dev);

	return 0;

err_release_clk:
	return ret;
}

static int vatics_i2s_remove(struct platform_device *pdev)
{
	VPL_DEBUG("%s +++\n", __func__);
	iounmap(g_sysc_base);
	iounmap(pllc_base);
	release_mem_region(SYSC_BASE,0x100);
	return 0;
}

static struct platform_driver vatics_i2s_driver = {
	.probe		= vatics_i2s_probe,
	.remove		= vatics_i2s_remove,
	.driver		= {
		.name	= VPL_PLATFORM,
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(vatics_i2s_dt_ids),
	},
};
//module_platform_driver(vatics_i2s_driver);

static int __init vatics_i2s_init(void)
{

	VPL_DEBUG("%s +++\n", __func__);
	platform_driver_register(&vatics_i2s_driver);
	platform_device_register_simple(VPL_PLATFORM, -1, NULL, 0);
	return 0;
}

static void __exit vatics_i2s_exit(void)
{
	VPL_DEBUG("%s +++\n", __func__);
}

module_init(vatics_i2s_init);
module_exit(vatics_i2s_exit);


MODULE_DESCRIPTION("VATICS Inc. vatics I2S SoC Interface");
MODULE_LICENSE("GPL");
