/*
 * Copyright (C) 2013  VATICS Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/module.h>
#include <linux/ioctl.h>
#include <linux/delay.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/err.h>
#include <linux/slab.h>
#include <asm/cacheflush.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/irqdomain.h>
#include <linux/dma-mapping.h>

#include <sound/core.h>
#include <sound/soc.h>

#include "vatics-asoc.h"

#define VAUDIO_DRIVER_VERSION		"1.0.0.0"
#define VAUDIO_COMPANY			"VATICS Inc."
#define VAUDIO_PLATFORM			"EVM"
const char vaudio_id[] = "$Vaudio: "VAUDIO_DRIVER_VERSION" "VAUDIO_PLATFORM" "__DATE__" "VAUDIO_COMPANY" $";

static struct dma_regs_t *vpl_audio_apb_descbuf_playback ;
static struct dma_regs_t *vpl_audio_apb_descbuf_capture ;

static struct snd_pcm_hardware vatics_pcm_hardware =
{
	.info =		 (SNDRV_PCM_INFO_INTERLEAVED | SNDRV_PCM_INFO_NONINTERLEAVED |
				   SNDRV_PCM_INFO_BLOCK_TRANSFER |
				   SNDRV_PCM_INFO_MMAP | SNDRV_PCM_INFO_MMAP_VALID),
	.formats =	SNDRV_PCM_FMTBIT_S16_LE,
	.rates =		(SNDRV_PCM_RATE_8000 | SNDRV_PCM_RATE_16000 |\
                                   SNDRV_PCM_RATE_32000 |SNDRV_PCM_RATE_44100 |\
                                   SNDRV_PCM_RATE_96000 |\
                                   SNDRV_PCM_RATE_48000 |SNDRV_PCM_RATE_KNOT),
	.rate_min =		8000,
	.rate_max =		96000,
	.channels_min =		2,
	.channels_max =		2,
	.periods_min =		2,
	.periods_max =		8192,
	.buffer_bytes_max =	1024 * 1024,
	.period_bytes_min =	128,
	.period_bytes_max =	PERIOD_BYTES_MAX,
};

static int dma_req_sel[2] = {
	0x3, 0x4
};

static irqreturn_t audio_dma_isr(int irq, void *dev_id);

void vatics_read_dma(struct dma_regs_t* dma_mmr, int chn)
{
	dma_mmr->dwSrc_Addr = VATICS_APBC_READL(APBC_DMA_SRC_ADDR(chn));
	dma_mmr->dwDes_Addr = VATICS_APBC_READL(APBC_DMA_DES_ADDR(chn));
	dma_mmr->dwLLP_Addr = VATICS_APBC_READL(APBC_DMA_LLP(chn));
	dma_mmr->dwCtrl = VATICS_APBC_READL(APBC_DMA_CTRL(chn));
}

void vatics_write_dma(struct dma_regs_t* dma_mmr, int chn)
{
	VATICS_APBC_WRITEL(APBC_DMA_SRC_ADDR(chn), dma_mmr->dwSrc_Addr);
	VATICS_APBC_WRITEL(APBC_DMA_DES_ADDR(chn), dma_mmr->dwDes_Addr);
	VATICS_APBC_WRITEL(APBC_DMA_LLP(chn), dma_mmr->dwLLP_Addr);
	VATICS_APBC_WRITEL(APBC_DMA_CTRL(chn), dma_mmr->dwCtrl);
}

void vatics_disable_dma(int dma_chn)
{
	unsigned long reg;

	reg = VATICS_APBC_READL(APBC_DMA_CTRL(dma_chn));
	reg &= ~0x3;
	VATICS_APBC_WRITEL(APBC_DMA_CTRL(dma_chn), reg);
	reg = VATICS_APBC_READL(APBC_DMA_CHN_MONITOR);
	reg &= ~(1<<dma_chn);
	VATICS_APBC_WRITEL(APBC_DMA_CHN_MONITOR, reg);
}

static void vatics_dma_getposition(int chn, dma_addr_t* src, dma_addr_t* dst)
{
	*src = (dma_addr_t)(VATICS_APBC_READL(APBC_DMA_SRC_ADDR(chn)));
	*dst = (dma_addr_t)(VATICS_APBC_READL(APBC_DMA_DES_ADDR(chn)));
}


static void vatics_pcm_start_dma(struct snd_pcm_substream *substream)
{
	struct vatics_runtime_data *prtd = substream->runtime->private_data;
	struct snd_pcm_runtime *runtime = substream->runtime;
	int dev_num = prtd->dev_num;
	unsigned int dma_size;
	unsigned int index;
	unsigned int offset;
	struct dma_regs_t *desc_addr = prtd->dma_data->desc_addr;
	struct dma_regs_t dma_mmr_init;

	VPL_DEBUG("%s +++\n", __func__);
	dma_size = snd_pcm_lib_period_bytes(substream); // transfer one period size data each dma move
	memset(&dma_mmr_init, 0, sizeof(dma_mmr_init));

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		dma_mmr_init.dwSrc_Addr = runtime->dma_addr;
		dma_mmr_init.dwDes_Addr = ssp_tx[dev_num] + I2S_TXDMA;
		dma_mmr_init.dwLLP_Addr = virt_to_phys(desc_addr);
		dma_mmr_init.dwCtrl = 0x50159 |
							(dma_req_sel[1] << 12) |
							((dma_size / 8) << 20);
		for (index = 0; index < (runtime->periods); index++) {
			offset = dma_size * ((index + 1) % runtime->periods);
			desc_addr[index].dwSrc_Addr = dma_mmr_init.dwSrc_Addr + offset;
			desc_addr[index].dwDes_Addr = dma_mmr_init.dwDes_Addr;
			desc_addr[index].dwLLP_Addr = dma_mmr_init.dwLLP_Addr + sizeof(struct dma_regs_t) * ((index + 1) % runtime->periods);
			desc_addr[index].dwCtrl = dma_mmr_init.dwCtrl;
		}
	}
	else {
		dma_mmr_init.dwSrc_Addr = ssp_rx[dev_num] + I2S_RXDMA;
		dma_mmr_init.dwDes_Addr = runtime->dma_addr;
		dma_mmr_init.dwLLP_Addr = virt_to_phys(desc_addr);
		dma_mmr_init.dwCtrl = 0x50499 |
							(dma_req_sel[0] << 12) |
							((dma_size / 8) << 20);
		for (index = 0; index < (runtime->periods); index++) {
			offset = dma_size * ((index + 1) % runtime->periods);
			desc_addr[index].dwSrc_Addr = dma_mmr_init.dwSrc_Addr;
			desc_addr[index].dwDes_Addr = dma_mmr_init.dwDes_Addr + offset;
			desc_addr[index].dwLLP_Addr = dma_mmr_init.dwLLP_Addr + sizeof(struct dma_regs_t) * ((index + 1) % runtime->periods);
			desc_addr[index].dwCtrl = dma_mmr_init.dwCtrl;
		}
	}
	/* XXX , please help to review it if the functiont is correct*/
	for (index = 0; index < (runtime->periods); index++) {
		flush_kernel_dcache_page(virt_to_page(&desc_addr[index]));
	}
	vatics_write_dma(&dma_mmr_init, prtd->dma_data->chn);
}

static void vatics_pcm_stop_dma(struct snd_pcm_substream *substream)
{
	struct vatics_runtime_data *prtd = substream->runtime->private_data;
	int dma_chn = prtd->dma_data->chn;

	VPL_DEBUG("%s +++\n", __func__);
	vatics_disable_dma(dma_chn);
}

static irqreturn_t audio_dma_isr(int irq, void *dev_id)
{
	struct snd_pcm_substream *substream = dev_id;
	struct vatics_runtime_data *prtd = substream->runtime->private_data;
	unsigned long chn_monitor;
	int index;

	chn_monitor = VATICS_APBC_READL(APBC_DMA_CHN_MONITOR);
	for (index = VATICS_I2S0; index < VATICS_I2S_NUM; index++) {
		if ((1<<(prtd->dma_data->chn)) & chn_monitor) {
			VATICS_APBC_WRITEL(APBC_DMA_CHN_MONITOR, ~(1<<(prtd->dma_data->chn)));
			snd_pcm_period_elapsed(substream);
			return IRQ_HANDLED;
		}
		if ((1<<(prtd->dma_data->chn)) & chn_monitor) {
			VATICS_APBC_WRITEL(APBC_DMA_CHN_MONITOR, ~(1<<(prtd->dma_data->chn)));	
			snd_pcm_period_elapsed(substream);
			return IRQ_HANDLED;
		}
	}
	return IRQ_NONE;
}

static int vatics_dma_irqrequest(struct snd_pcm_substream *substream)
{
	struct vatics_runtime_data *prtd = substream->runtime->private_data;
	struct vatics_pcm_dma_params *dma_data = prtd->dma_data;

	VPL_DEBUG("%s +++\n", __func__);
	spin_lock(&prtd->lock);
	dma_data->irq_acks++;
	if (dma_data->irq_acks == 1) {
		spin_unlock(&prtd->lock);
		dma_data->irq = irq_create_mapping(NULL, APBC_IRQ_NUM);
		if(request_irq (dma_data->irq, audio_dma_isr, IRQF_SHARED, "APBC DMA", substream)) {
			VPL_DEBUG( "irq request fail\n");
			spin_lock(&prtd->lock);
			dma_data->irq_acks--;
			spin_unlock(&prtd->lock);
			return -EBUSY;
		}
		// set Level Trigger for LLP interrupts
		irq_set_irq_type(dma_data->irq, IRQ_TYPE_LEVEL_HIGH);
		return 0;
	}
	spin_unlock(&prtd->lock);
	return 0;
}

static void vatics_dma_irqfree(struct snd_pcm_substream *substream)
{
	struct vatics_runtime_data *prtd = substream->runtime->private_data;
	struct vatics_pcm_dma_params *dma_data = prtd->dma_data;

	VPL_DEBUG("%s +++\n", __func__);
	spin_lock(&prtd->lock);
	if (dma_data->irq_acks > 1) {
		dma_data->irq_acks--;
	}
	else {
		free_irq(dma_data->irq, substream);
	}
	spin_unlock(&prtd->lock);
}

static int vatics_pcm_trigger(struct snd_pcm_substream *substream, int cmd)
{
	struct vatics_runtime_data *prtd = substream->runtime->private_data;
	int ret = 0;

	VPL_DEBUG("%s +++\n", __func__);
	spin_lock(&prtd->lock);
	switch (cmd) {
	case SNDRV_PCM_TRIGGER_START:
		vatics_pcm_start_dma(substream);
		break;

	case SNDRV_PCM_TRIGGER_STOP:
		vatics_pcm_stop_dma(substream);
		break;
	default:
		ret = -EINVAL;
		break;
	}
	spin_unlock(&prtd->lock);
	return ret;
}

static int vatics_pcm_prepare(struct snd_pcm_substream *substream)
{
	struct vatics_runtime_data *prtd = substream->runtime->private_data;
	struct vatics_pcm_dma_params *dma_data = prtd->dma_data;

	VPL_DEBUG("%s +++\n", __func__);
	/*dma_data->desc_addr = vpl_audio_apb_descbuf;
	if (dma_data->desc_addr == NULL) {
		VPL_DEBUG("No memory.\n");
		return -ENOMEM;
	}*/
	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		dma_data->desc_addr = vpl_audio_apb_descbuf_playback;
		prtd->dma_data->chn = DMA_CHN_8;
	}
	else {
		dma_data->desc_addr = vpl_audio_apb_descbuf_capture;
		prtd->dma_data->chn = DMA_CHN_6;
	}
	return 0;
}

static snd_pcm_uframes_t vatics_pcm_pointer(struct snd_pcm_substream *substream)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	struct vatics_runtime_data *prtd = runtime->private_data;
	unsigned int offset;
	dma_addr_t count;
	dma_addr_t src, dst;

	spin_lock(&prtd->lock);
	vatics_dma_getposition(prtd->dma_data->chn, &src, &dst);
	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK)
		count = src - runtime->dma_addr;
	else
		count = dst - runtime->dma_addr;;

	spin_unlock(&prtd->lock);

	offset = bytes_to_frames(runtime, count);
	if (offset >= runtime->buffer_size)
		offset = 0;

	return offset;
}

static int vatics_pcm_open(struct snd_pcm_substream *substream)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	struct vatics_runtime_data *prtd;
	struct vatics_pcm_dma_params *dma_data;
	int ret = 0;

	VPL_DEBUG("%s +++\n", __func__);
	snd_soc_set_runtime_hwparams(substream, &vatics_pcm_hardware);

	prtd = kzalloc(sizeof(*prtd), GFP_KERNEL);
	if (prtd == NULL)
		return -ENOMEM;
	dma_data = kzalloc(sizeof(*dma_data), GFP_KERNEL);
	if (dma_data == NULL)
		return -ENOMEM;

	spin_lock_init(&prtd->lock);
	prtd->dev_num = VATICS_I2S0;
	prtd->dma_data = dma_data;
	runtime->private_data = prtd;
	vatics_dma_irqrequest(substream);

	return ret;
}

static int vatics_pcm_close(struct snd_pcm_substream *substream)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	struct vatics_runtime_data *prtd = runtime->private_data;

	VPL_DEBUG("%s +++\n", __func__);
	vatics_dma_irqfree(substream);
	kfree(prtd);
	return 0;
}

static int vatics_pcm_hw_params(struct snd_pcm_substream *substream, struct snd_pcm_hw_params *hw_params)
{
	VPL_DEBUG("%s +++\n", __func__);
	return snd_pcm_lib_malloc_pages(substream, params_buffer_bytes(hw_params));
}

static int vatics_pcm_hw_free(struct snd_pcm_substream *substream)
{
	VPL_DEBUG("%s +++\n", __func__);
	return snd_pcm_lib_free_pages(substream);
}

struct snd_pcm_ops vatics_pcm_ops = {
	.open = vatics_pcm_open,
	.close = vatics_pcm_close,
	.ioctl = snd_pcm_lib_ioctl,
	.hw_params = vatics_pcm_hw_params,
	.hw_free = vatics_pcm_hw_free,
	.prepare = vatics_pcm_prepare,
	.trigger = vatics_pcm_trigger,
	.pointer = vatics_pcm_pointer,
};

static int vatics_pcm_preallocate_dma_buffer(struct snd_pcm *pcm, int stream)
{
	struct snd_pcm_substream *substream = pcm->streams[stream].substream;
	struct snd_dma_buffer *buf = &substream->dma_buffer;
	size_t size = vatics_pcm_hardware.buffer_bytes_max;

	VPL_DEBUG("%s +++\n", __func__);
	buf->dev.type = SNDRV_DMA_TYPE_DEV;
	buf->dev.dev = pcm->card->dev;
	buf->private_data = NULL;
	buf->area = dma_alloc_coherent(pcm->card->dev, size, &buf->addr, GFP_KERNEL);
	if (!buf->area)
		return -ENOMEM;

	buf->bytes = size;
	return 0;
}

static void vatics_pcm_free_dma_buffers(struct snd_pcm *pcm)
{
	struct snd_pcm_substream *substream;
	struct snd_dma_buffer *buf;
	int stream;

	VPL_DEBUG("%s +++\n", __func__);
	for (stream = 0; stream < 2; stream++) {
		substream = pcm->streams[stream].substream;
		if (!substream)
			continue;

		buf = &substream->dma_buffer;
		if (!buf->area)
			continue;
		dma_free_coherent(pcm->card->dev, buf->bytes, buf->area, buf->addr);
		buf->area = NULL;
	}
}

static u64 vatics_pcm_dmamask = 0xffffffff;

static int vatics_pcm_new(struct snd_soc_pcm_runtime *rtd)
{
	int ret;
	struct snd_card *card = rtd->card->snd_card;
	struct snd_pcm *pcm = rtd->pcm;
	VPL_DEBUG("%s +++\n", __func__);

	if (!card->dev->dma_mask)
		card->dev->dma_mask = &vatics_pcm_dmamask;
	if (!card->dev->coherent_dma_mask)
		card->dev->coherent_dma_mask = 0xffffffff;

	if (pcm->streams[SNDRV_PCM_STREAM_PLAYBACK].substream) {
		ret = vatics_pcm_preallocate_dma_buffer(pcm,
					SNDRV_PCM_STREAM_PLAYBACK);
		if (ret) {
			VPL_DEBUG("vatics_pcm_new:0\n");
			return ret;
		}
	}

	if (pcm->streams[SNDRV_PCM_STREAM_CAPTURE].substream) {
		ret = vatics_pcm_preallocate_dma_buffer(pcm, SNDRV_PCM_STREAM_CAPTURE);
		if (ret) {
			VPL_DEBUG("vatics_pcm_new:1\n");
			return ret;
		}
	}

	vpl_audio_apb_descbuf_capture = \
		(struct dma_regs_t *)kmalloc((vatics_pcm_hardware.periods_max) * sizeof(struct dma_regs_t), GFP_KERNEL) ;
	vpl_audio_apb_descbuf_playback = \
		(struct dma_regs_t *)kmalloc((vatics_pcm_hardware.periods_max) * sizeof(struct dma_regs_t), GFP_KERNEL) ;	

	return 0;
}

struct snd_soc_platform_driver vatics_soc_platform = {
	.ops = &vatics_pcm_ops,
	.pcm_new = vatics_pcm_new,
	.pcm_free = vatics_pcm_free_dma_buffers,
};

int vatics_soc_platform_register(struct device *dev)
{
	VPL_DEBUG("%s +++\n", __func__);
	return snd_soc_register_platform(dev, &vatics_soc_platform);
}
EXPORT_SYMBOL(vatics_soc_platform_register);

void vatics_soc_platform_unregister(struct device *dev)
{
	VPL_DEBUG("%s +++\n", __func__);
	snd_soc_unregister_platform(dev);
}
EXPORT_SYMBOL(vatics_soc_platform_unregister);

MODULE_DESCRIPTION("VATICS Inc. vatics PCM DMA module");
MODULE_LICENSE("GPL");
