/*
 *
 * Copyright (C) 2014  VATICS Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/device.h>
#include <linux/delay.h>
#include <linux/io.h>
#include <linux/clk.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include <linux/completion.h>

#include <linux/of.h>

#include "vivo_audio.h"
#include "vatics-algo.h"

static struct vatics_algo va_algo;
void register_algo(struct vatics_algo *algo)
{
	va_algo.algo_init = algo->algo_init;
	va_algo.algo_handle = algo->algo_handle;
	va_algo.process_size = algo->process_size;
}
EXPORT_SYMBOL(register_algo);
void unregister_algo(void)
{
	va_algo.algo_init = NULL;
	va_algo.algo_handle = NULL;
}
EXPORT_SYMBOL(unregister_algo);
char* ref_buf;
int vivo_trigger(struct vivo_audio_dev *adev, int cmd, int direction);

#ifdef DEBUG_APP
/* NOTE: please be noticed!!! the debug app interface is for debug/demo purpose */
static int audio_open(struct inode *inode, struct file *filp)
{
	struct vivo_audio_dev *adev;
	adev = container_of((struct cdev*)inode->i_cdev, struct vivo_audio_dev, cdev);
	filp->private_data = adev;
	printk("%s(%d) adev:period:%d\n", __func__, __LINE__, adev->runtime_periods);
	return 0;
}

static int audio_release(struct inode *inode, struct file *filp)
{
	printk("%s(%d)\n", __func__, __LINE__);
	return 0;
}

static int start = 0;
static long audio_ioctl (struct file *filp, unsigned int cmd,
		unsigned long arg)
{
	struct vivo_audio_dev *adev = filp->private_data;
	printk("%s(%d) cmd:%u\n", __func__, __LINE__, cmd);
	switch (cmd) {
		case 10:
			vivo_trigger(adev, AUDIO_TRIGGER_STOP, AUDIO_STREAM_PLAYBACK);
			break;
		case 11:
			vivo_trigger(adev, AUDIO_TRIGGER_START, AUDIO_STREAM_PLAYBACK);
			start = 1;
			break;
		case 20:
			if (va_algo.algo_init) {
				ret = va_algo.algo_init(adev->sample_rate);
				if (ret) {
					pr_warning("algo init failed!\n");
				}
			}
			vivo_trigger(adev, AUDIO_TRIGGER_STOP, AUDIO_STREAM_CAPTURE);
			break;
		case 21:
			vivo_trigger(adev, AUDIO_TRIGGER_START, AUDIO_STREAM_CAPTURE);
			break;
	}
	return 0;
}

#define PLAYBACK_STOP_THRESHOLD    128  //if available < threshold, we stop input data
#define PLAYBACK_RESTART_THRESHOLD 400  //if available > threshold, restart input data
DECLARE_COMPLETION(playback_complete);
DECLARE_COMPLETION(capture_complete);
static atomic_t playback_input_stop = ATOMIC_INIT(0);

inline unsigned int playback_available(struct vivo_dma_buffer* dma_buffer)
{
	unsigned int app_off, hw_off, offset;
	struct vivo_audio_dev *adev = dma_buffer->adev;
	app_off = addr_to_period_num(dma_buffer);
	hw_off  = vivo_pcm_pointer(adev, AUDIO_STREAM_PLAYBACK);
//	printk("app_off:%u hw_off:%u\n", app_off, hw_off);
	if (hw_off >= app_off)
		offset = hw_off - app_off;
	else
		offset = hw_off + dma_buffer->adev->runtime_periods - app_off;
	return offset;
}

/* this is an example for getting app/hw period base address */
inline void print_debug_address(struct vivo_dma_buffer* dma_buffer)
{
	struct vivo_audio_dev *adev = dma_buffer->adev;
	unsigned int app_off = addr_to_period_num(dma_buffer);
	void* app_addr = period_num_to_addr(dma_buffer, app_off);
	unsigned int hw_off = vivo_pcm_pointer(adev, AUDIO_STREAM_PLAYBACK);
	void* hw_addr = period_num_to_addr(dma_buffer, hw_off);

	printk("app_off:%u, hw_off:%u app_idx:%lx\n", app_off, hw_off, 
					(long)dma_buffer->app_idx);
	printk("app_addr:%p hw_addr:%p\n", app_addr, hw_addr);
}

ssize_t audio_write(struct file *filp, const char __user *buf,
					size_t count, loff_t *f_pos)
{
	ssize_t size;
	struct vivo_audio_dev *adev = filp->private_data;
	struct vivo_dma_buffer* dma_buffer = adev->dma_buffer[AUDIO_STREAM_PLAYBACK];
	int total_size = adev->runtime_periods * adev->runtime_dmasize;
	long boundary = (long)dma_buffer->area + total_size;
	unsigned int avail;
	/*  XXX the path is only for demo/debug purpose, we ignore the short write case
	 *      we assume every write length is the same as the period size.
	 *      the users MUST handle non-matching cases themselves.
	 */
	size = copy_from_user(dma_buffer->app_idx, buf, count);

	/* spin lock the app_idx ? */
	dma_buffer->app_idx += count;

	if (!start)
		return count;
	avail = playback_available(dma_buffer);

	if (avail < PLAYBACK_STOP_THRESHOLD) {
		print_debug_address(dma_buffer);
		VPL_DEBUG("p:stop avail:%u\n", avail)
		atomic_set(&playback_input_stop, 1);
		wait_for_completion(&playback_complete);
	}

	if ((long)dma_buffer->app_idx -  boundary >= 0) {
		VPL_DEBUG("dma_buffer->app_idx:0x%lx\n", (long)dma_buffer->app_idx);
		VPL_DEBUG("boundary:0x%lx\n", boundary);
		dma_buffer->app_idx = dma_buffer->area;
	}
	return count;
}

ssize_t audio_read(struct file *filp, char __user *buf,
					size_t count, loff_t *f_pos)
{
	ssize_t size;
	struct vivo_audio_dev *adev = filp->private_data;
	struct vivo_dma_buffer* dma_buffer = adev->dma_buffer[AUDIO_STREAM_CAPTURE];
	int total_size = adev->runtime_periods * adev->runtime_dmasize;
	long boundary = (long)dma_buffer->area + total_size;

	wait_for_completion(&capture_complete);
	if (va_algo.algo_handle) {
		/*  add count and process_size align check
		 *  we assume the count aligns with process_size
		 */
		for(i = 0 ; i < count / va_algo.process_size ; i++)
			va_algo.algo_handle(dma_buffer->app_idx +
						va_algo.process_size * i,
						ref_buf + va_algo.process_size * i);
	}
	size = copy_to_user(buf, dma_buffer->app_idx, count);
	dma_buffer->app_idx = dma_buffer->app_idx + count;

	if ((long)dma_buffer->app_idx -  boundary >= 0) {
		VPL_DEBUG("dma_buffer->app_idx:0x%lx\n", (long)dma_buffer->app_idx);
		VPL_DEBUG("boundary:0x%lx\n", boundary);
		dma_buffer->app_idx = dma_buffer->area;
	}
	return count;
}

static struct file_operations audio_fops = {
		open:           audio_open,
		write:          audio_write,
		read:           audio_read,
		unlocked_ioctl: audio_ioctl,
		release:        audio_release,
};

static int playback_notify_call(struct notifier_block *this,
                            unsigned long code, void *data)
{
	struct vivo_audio_dev *adev = data;
	struct vivo_dma_buffer* dma_buffer = adev->dma_buffer[AUDIO_STREAM_PLAYBACK];
	unsigned int avail;
	/* TODO maybe we can clean previous data here in case underrun */

	avail = playback_available(dma_buffer);

	if (avail > PLAYBACK_RESTART_THRESHOLD && atomic_read(&playback_input_stop)) {
		VPL_DEBUG("p:restart avail:%u\n", avail);
		atomic_set(&playback_input_stop, 0);
		complete(&playback_complete);
	}
	return 0;
}

static struct notifier_block playback_notify =
{
	.notifier_call = playback_notify_call,
};

static int capture_notify_call(struct notifier_block *this,
                            unsigned long code, void *data)
{
	//printk("%s +++ \n", __func__);
	complete(&capture_complete);

	return 0;
}

static struct notifier_block capture_notify =
{
	.notifier_call = capture_notify_call,
};
#endif
#ifdef DEBUG_LOOPBACK 
/* capture to playback directly */
static int capture_loopback_call(struct notifier_block *this,
                            unsigned long code, void *data)
{
// get first frame, put it to playback frame, and start playback
	struct vivo_audio_dev *adev = data;
	struct vivo_dma_buffer *cap_buf = adev->dma_buffer[AUDIO_STREAM_CAPTURE];
	struct vivo_dma_buffer *play_buf = adev->dma_buffer[AUDIO_STREAM_PLAYBACK];
	int size = adev->runtime_dmasize;
	int total_size = adev->runtime_periods * adev->runtime_dmasize;
	long boundary = (long)cap_buf->area + total_size;
	int i;
	/* NOTE:we need to get some data from other source */
	if (va_algo.algo_handle) {
		for(i = 0 ; i < size / va_algo.process_size ; i++)
			va_algo.algo_handle(cap_buf->app_idx + 
						va_algo.process_size * i,
						ref_buf + va_algo.process_size * i);
	}
	memcpy(play_buf->app_idx, cap_buf->app_idx, size);
	cap_buf->app_idx = cap_buf->app_idx + size;
	play_buf->app_idx = play_buf->app_idx + size;

	/* zeroing index */
	if ((long)cap_buf->app_idx -  boundary >= 0) {
		cap_buf->app_idx = cap_buf->area;
		play_buf->app_idx = play_buf->area;
	}

	return 0;
}

static struct notifier_block capture_loopback =
{
	.notifier_call = capture_loopback_call,
};

static int start = 0;
static ssize_t vpl_audio_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	return sprintf(buf, "%d\n", start);
}
static ssize_t vpl_audio_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t size)
{
	struct vivo_audio_dev *adev = dev_get_drvdata(dev);
	struct vivo_dma_buffer *cap_buf = adev->dma_buffer[AUDIO_STREAM_CAPTURE];
	struct vivo_dma_buffer *play_buf = adev->dma_buffer[AUDIO_STREAM_PLAYBACK];
	int ret;
	start = simple_strtol(buf, NULL, 10);
	if (start) {
		if (va_algo.algo_init) {
			ret = va_algo.algo_init(adev->sample_rate);
			if (ret) {
				pr_warning("algo init failed!\n");
			}
			if (adev->runtime_dmasize % va_algo.process_size != 0)
				pr_warning("non-align algo process size, please check!!\n");
		}
		vivo_trigger(adev, AUDIO_TRIGGER_START, AUDIO_STREAM_CAPTURE);
		vivo_trigger(adev, AUDIO_TRIGGER_START, AUDIO_STREAM_PLAYBACK);
		start = 1;
	} else {
		vivo_trigger(adev, AUDIO_TRIGGER_STOP, AUDIO_STREAM_CAPTURE);
		vivo_trigger(adev, AUDIO_TRIGGER_STOP, AUDIO_STREAM_PLAYBACK);
		start = 0;
		/* user need to reset the app_idx to zero */
		cap_buf->app_idx = cap_buf->area;
		play_buf->app_idx = play_buf->area;
	}
	return size;
}


static /*const*/ DEVICE_ATTR(start, 0644,
                vpl_audio_show, vpl_audio_store);

#endif

int vivo_trigger(struct vivo_audio_dev *adev, int cmd, int direction)
{
	vivo_codec_trigger(cmd, direction);
	vivo_pcm_trigger(adev, cmd, direction);
	vivo_i2s_trigger(cmd, direction);
	return 0;
}

static int vivo_audio_probe(struct platform_device *pdev)
{
	__maybe_unused int ret, major;
	__maybe_unused dev_t dev_no;
	__maybe_unused static struct class *class;
	struct vivo_audio_dev *adev;
	VPL_DEBUG("%s +++\n", __func__);

	adev = devm_kzalloc(&pdev->dev, sizeof(struct vivo_audio_dev),
			   GFP_KERNEL);
	adev->dev = &pdev->dev;

	request_mem_region(VPL_SYSC_MMR_BASE, 0x100, "SYSC"); 
	adev->sysc_base = ioremap(VPL_SYSC_MMR_BASE, 0x100);
	request_mem_region(VPL_PLLC_MMR_BASE, 0x100, "PLLC"); 
	adev->pllc_base = ioremap(VPL_PLLC_MMR_BASE, 0x100);
	request_mem_region(VPL_ACDCC_MMR_BASE, 0x100, "ACDCC"); 
	adev->dwc_base  = ioremap(VPL_ACDCC_MMR_BASE,0x100);

	platform_set_drvdata(pdev, adev);
#ifdef DEBUG_APP
	cdev_init(&adev->cdev, &audio_fops);
	adev->cdev.owner = THIS_MODULE;
	adev->cdev.ops = &audio_fops;
	ret = alloc_chrdev_region(&dev_no, 0, 1, "audio");
	if (ret < 0) {
		dev_err(adev->dev, "allocate major number failed!");
		return -ENODEV;
	}
	major = MAJOR(dev_no);
	ret = cdev_add(&adev->cdev, dev_no, 1);
	if (ret < 0) {
		dev_err(adev->dev, "add cdev failed!");
		return -ENODEV;
	}

	class = class_create(THIS_MODULE, "audio");
	device_create(class, NULL, MKDEV(major, 0), NULL, "audio");
#endif

	vivo_i2s_init(adev);
	vivo_codec_init(adev);

	/*  data depth 16bit = 2bytes
	 *  2-channel, 2*2 = 4 byte
	 *  10ms, handle cycle, 1000 / 10 = 100
	 *  16000 / 100 = 160, need to handle 160 sample per cycle
	 *  160 * 4 = 640, 640 bytes transfer we interrupt one time
	 */
	// TODO export function to set the period, we hard code here first
	adev->runtime_periods = 512*2;
	//adev->runtime_periods = 512;
	//adev->runtime_dmasize = 0x800; //2048
	adev->runtime_dmasize = 640;

	vivo_pcm_new(adev, AUDIO_STREAM_PLAYBACK);
	vivo_pcm_new(adev, AUDIO_STREAM_CAPTURE);

	//TODO export index control function
	adev->dma_buffer[AUDIO_STREAM_PLAYBACK]->app_idx
			= adev->dma_buffer[AUDIO_STREAM_PLAYBACK]->area;

	adev->dma_buffer[AUDIO_STREAM_CAPTURE]->app_idx
			= adev->dma_buffer[AUDIO_STREAM_CAPTURE]->area;
	adev->sample_rate = 16000;
	//vivo_codec_set_clkdiv(48000);
	vivo_codec_set_clkdiv(adev->sample_rate);
#ifdef DEBUG_APP
	register_playback_notifier(&playback_notify);
	register_capture_notifier(&capture_notify);
#endif
#ifdef DEBUG_LOOPBACK
	register_capture_notifier(&capture_loopback);
	device_create_file(&pdev->dev, &dev_attr_start);
#endif
	return 0;
}

static int vivo_audio_remove(struct platform_device *pdev)
{
	struct vivo_audio_dev *adev = dev_get_drvdata(&pdev->dev);
	VPL_DEBUG("%s +++\n", __func__);
	iounmap(adev->sysc_base);
	iounmap(adev->pllc_base);
	vivo_pcm_close(adev);
	return 0;
}

static const struct of_device_id vivo_audio_dt_ids[] = {
	{
		.compatible = "vivo,vpl-audio",
	},
	{ /* sentinel */ }
};
MODULE_DEVICE_TABLE(of, vivo_audio_dt_ids);

static struct platform_driver vivo_audio_driver = {
	.probe		= vivo_audio_probe,
	.remove		= vivo_audio_remove,
	.driver		= {
		.name	= "vivo_audio",
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(vivo_audio_dt_ids),
	},
};
//module_platform_driver(vivo_audio_driver);

static int __init vivo_audio_init(void)
{

	VPL_DEBUG("%s +++\n", __func__);
	platform_driver_register(&vivo_audio_driver);
	platform_device_register_simple("vivo_audio", -1, NULL, 0);
	return 0;
}

static void __exit vivo_audio_exit(void)
{
	VPL_DEBUG("%s +++\n", __func__);
	platform_driver_unregister(&vivo_audio_driver);
}

module_init(vivo_audio_init);
module_exit(vivo_audio_exit);

