/*
 *
 * Copyright (C) 2014  VATICS Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/device.h>
#include <linux/delay.h>
#include <linux/io.h>
#include <linux/clk.h>

#include <linux/of.h>

#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/initval.h>
#include <sound/soc.h>

#include "vivo_audio.h"

// SYSC
#define SYSC_ID_VERSION       			0x00000000
#define SYSC_I2S_CLK_PARAM 			0x0000002C
#define SYSC_I2S_CLK_ENABLE			0x00000028
#define SYSC_I2S_ENABLE				0x00000800
#define SYSC_IF_CTRL                    	0x00000048

// register offset
#define SYSC_PAD_EN_CTRL_5                      0x000000B0
#define SYSC_I2SSC_CTRL_0                       0x00000058
#define SYSC_I2SSC_CTRL_1                       0x0000005C

// SYSC_I2SSC_CTRL_0, shift
#define I2SSC_TX_WS_DIV                         24
#define I2SSC_TX_SCLK_DIV                       16
#define I2SSC_TX_MCLK_DIV                       8
#define I2SSC_RX_MCLK_DIV                       0
// SYSC_I2SSC_CTRL_1, shift
#define I2SSC_MCLK_PAD_OUT_DRV_CTRL             20
#define I2SSC_PAD_OUT_DRV_CTRL                  16
#define I2SSC_MCLK_PAD_SR_CTRL                  12
#define I2SSC_TX_DST_SRC_SEL                    8
#define I2SSC_TX_SCLK_SRC_SEL                   4
#define I2SSC_RX_SCLK_SRC_SEL                   0

#define PLLC_5_CTRL                             0x2C
#define PLLC_5_DIV                              0x30

static void *g_sysc_base;
static void *pllc_base;

void set_pll5(int target)
{
	/* default, target = pll5 / mclk_div,
	 *                 = pll5 / 8
	 */
	u32 reg;
	switch (target) {
		// use one pll table here
		case 12000000:
			reg = 0x7cf0918;
			break;
		case 49152000:
			reg = 0x7ff1318;
			break;
		case 24576000:
			reg = 0x7ff1331;
			break;
		default:
			printk("unknown target, do nothing\n");
			return;
	}
	writel(reg, IOMEM((unsigned long)pllc_base + PLLC_5_DIV));

	reg = readl(IOMEM((unsigned long)pllc_base + PLLC_5_CTRL));
	reg |= 0x1;
	writel(reg, IOMEM((unsigned long)pllc_base + PLLC_5_CTRL));
	while (readl(IOMEM((unsigned long)pllc_base + PLLC_5_CTRL)) != 0x30) {
		printk("Unlock....\n");
	}

}

static void __i2s_init(void)
{
	/* Initialize I2S */
	unsigned int dev_tx = ssp_tx[0];
	unsigned int dev_rx = ssp_rx[0];
	unsigned int virt_dev_tx = __IO_ADDRESS(dev_tx);
	unsigned int virt_dev_rx = __IO_ADDRESS(dev_rx);
	u32 reg = 0;
	static int isInit = 1;
	struct clk *clk;

	/* init the mclk output for codec*/
	if (isInit) {

		reg = readl(IOMEM((unsigned long)g_sysc_base + SYSC_PAD_EN_CTRL_5));
//#define EXTERNAL
//#define MASTER
#ifdef EXTERNAL
		reg|= 1<<10;    //I2SSC_TXPAD_PAD_EN
		reg|= 1<<9;     //I2SSC_RXPAD_PAD_EN
		reg&= ~(1<<20); //N903U PAD EN, turn off (share pin with i2ss external)
#ifdef MASTER
		reg|= 1<<12;  //I2SSC_MASTER
#endif
		reg|= 1<<11;  //I2SSC_MCLK_PAD_EN
#endif
		writel(reg, IOMEM((unsigned long)g_sysc_base + SYSC_PAD_EN_CTRL_5));

		/*pll5 set divider to 12.288M */
		//reg = 0x7000101;
		//writel(reg, IOMEM((unsigned long)g_sysc_base + SYSC_I2SSC_CTRL_0));

		clk = clk_get(NULL, "vpl_i2ssc");
		if (IS_ERR(clk)) {
			printk(KERN_ERR "[I2S] Failed to get clock\n");
			return;
		}
		clk_prepare_enable(clk);

		isInit = 0;
	}
	/* set Tx/Rx clock source, data destination*/
#ifdef EXTERNAL
	reg = 0x0;
	reg|= 0x1<<I2SSC_TX_DST_SRC_SEL; // TX data DST, external
#ifdef MASTER
	reg|= 0x2<<I2SSC_TX_SCLK_SRC_SEL; // TX sclk source, from PLL
#else
	reg|= 0x1<<I2SSC_TX_SCLK_SRC_SEL; // TX sclk source, from external
#endif // end of MASTER
	reg|= 0x1; // RX sclk source, from external
#else
	// internal codec
	set_pll5(12000000);
	reg = 0x0;
#endif
	writel(reg, IOMEM((unsigned long)g_sysc_base + SYSC_I2SSC_CTRL_1));

	VATICS_VPL_SSP_WRITEL(virt_dev_tx, I2S_IER, 0x0);
	VATICS_VPL_SSP_WRITEL(virt_dev_tx, I2S_IRER, 0x0);
	VATICS_VPL_SSP_WRITEL(virt_dev_tx, I2S_ITER, 0x0);
	VATICS_VPL_SSP_WRITEL(virt_dev_tx, I2S_IMR, 0x0);
	VATICS_VPL_SSP_WRITEL(virt_dev_tx, I2S_RFCR, 0x3);
	VATICS_VPL_SSP_WRITEL(virt_dev_tx, I2S_TFCR, 0x3);
	VATICS_VPL_SSP_WRITEL(virt_dev_tx, I2S_RFF0, 0x1);
	VATICS_VPL_SSP_WRITEL(virt_dev_tx, I2S_TFF0, 0x1);
	VATICS_VPL_SSP_WRITEL(virt_dev_tx, I2S_RCR, 0x2);
	VATICS_VPL_SSP_WRITEL(virt_dev_tx, I2S_TCR, 0x2);

	VATICS_VPL_SSP_WRITEL(virt_dev_rx, I2S_IER, 0x0);
	VATICS_VPL_SSP_WRITEL(virt_dev_rx, I2S_IRER, 0x0);
	VATICS_VPL_SSP_WRITEL(virt_dev_rx, I2S_ITER, 0x0);
	VATICS_VPL_SSP_WRITEL(virt_dev_rx, I2S_IMR, 0x0);
	VATICS_VPL_SSP_WRITEL(virt_dev_rx, I2S_RFCR, 0x3);
	VATICS_VPL_SSP_WRITEL(virt_dev_rx, I2S_TFCR, 0x3);
	VATICS_VPL_SSP_WRITEL(virt_dev_rx, I2S_RFF0, 0x1);
	VATICS_VPL_SSP_WRITEL(virt_dev_rx, I2S_TFF0, 0x1);
	VATICS_VPL_SSP_WRITEL(virt_dev_rx, I2S_RCR, 0x2);
	VATICS_VPL_SSP_WRITEL(virt_dev_rx, I2S_TCR, 0x2);
}

static void vatics_i2s_enable_device(int dev_num, int enable, int stream_id)
{
	unsigned int dev_tx = ssp_tx[dev_num];
	unsigned int dev_rx = ssp_rx[dev_num];
	unsigned int virt_dev_tx = __IO_ADDRESS(dev_tx);
	unsigned int virt_dev_rx = __IO_ADDRESS(dev_rx);
	VPL_DEBUG("%s +++\n", __func__);
	if (stream_id == AUDIO_STREAM_PLAYBACK) {
		VATICS_VPL_SSP_WRITEL(virt_dev_tx, I2S_IER, (enable) ? 0x1 : 0x0);
	} else {
		VATICS_VPL_SSP_WRITEL(virt_dev_rx, I2S_IER, (enable) ? 0x1 : 0x0);
	}
}

static void vatics_i2s_enable_stream(int dev_num, int enable, int stream_id)
{
	unsigned int dev_tx = ssp_tx[dev_num];
	unsigned int dev_rx = ssp_rx[dev_num];
	unsigned int virt_dev_tx = __IO_ADDRESS(dev_tx);
	unsigned int virt_dev_rx = __IO_ADDRESS(dev_rx);

	VPL_DEBUG("%s +++\n", __func__);
	if (stream_id == AUDIO_STREAM_PLAYBACK) {
		VATICS_VPL_SSP_WRITEL(virt_dev_tx, I2S_ITER, (enable) ? 0x1 : 0x0);
	}
	else {
		VATICS_VPL_SSP_WRITEL(virt_dev_rx, I2S_IRER, (enable) ? 0x1 : 0x0);
	}
}

int vivo_i2s_trigger(int cmd, int direction)
{
	int ret = 0;

	VPL_DEBUG("%s +++\n", __func__);
	switch (cmd) {
	case AUDIO_TRIGGER_START:
		vatics_i2s_enable_device(0, 1, direction);
		vatics_i2s_enable_stream(0, 1, direction);
		break;

	case AUDIO_TRIGGER_STOP:
		vatics_i2s_enable_stream(0, 0, direction);
		break;
	default:
		ret = -EINVAL;
		break;
	}
	return 0;
}

int vivo_i2s_init(struct vivo_audio_dev* adev)
{
	g_sysc_base = adev->sysc_base;
	pllc_base   = adev->pllc_base;
	__i2s_init();
	return 0;
}
