/*
 * Copyright (C) 2013  VATICS Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/module.h>
#include <linux/ioctl.h>
#include <linux/delay.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/err.h>
#include <linux/slab.h>
#include <asm/cacheflush.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/irqdomain.h>
#include <linux/dma-mapping.h>

#include "vivo_audio.h"

BLOCKING_NOTIFIER_HEAD(playback_notifier_list);
BLOCKING_NOTIFIER_HEAD(capture_notifier_list);

int register_playback_notifier(struct notifier_block *nb)
{
	return blocking_notifier_chain_register(&playback_notifier_list, nb);
}
int register_capture_notifier(struct notifier_block *nb)
{
	return blocking_notifier_chain_register(&capture_notifier_list, nb);
}

static struct dma_regs_t *vpl_audio_apb_descbuf_playback ;
static struct dma_regs_t *vpl_audio_apb_descbuf_capture ;

static int dma_req_sel[2] = {
	0x3, 0x4
};

static irqreturn_t audio_dma_isr(int irq, void *dev_id);

void vatics_read_dma(struct dma_regs_t* dma_mmr, int chn)
{
	dma_mmr->dwSrc_Addr = VATICS_APBC_READL(APBC_DMA_SRC_ADDR(chn));
	dma_mmr->dwDes_Addr = VATICS_APBC_READL(APBC_DMA_DES_ADDR(chn));
	dma_mmr->dwLLP_Addr = VATICS_APBC_READL(APBC_DMA_LLP(chn));
	dma_mmr->dwCtrl = VATICS_APBC_READL(APBC_DMA_CTRL(chn));
}

void vatics_write_dma(struct dma_regs_t* dma_mmr, int chn)
{
	VATICS_APBC_WRITEL(APBC_DMA_SRC_ADDR(chn), dma_mmr->dwSrc_Addr);
	VATICS_APBC_WRITEL(APBC_DMA_DES_ADDR(chn), dma_mmr->dwDes_Addr);
	VATICS_APBC_WRITEL(APBC_DMA_LLP(chn), dma_mmr->dwLLP_Addr);
	VATICS_APBC_WRITEL(APBC_DMA_CTRL(chn), dma_mmr->dwCtrl);
}

void vatics_disable_dma(int dma_chn)
{
	unsigned long reg;

	reg = VATICS_APBC_READL(APBC_DMA_CTRL(dma_chn));
	reg &= ~0x3;
	VATICS_APBC_WRITEL(APBC_DMA_CTRL(dma_chn), reg);
	reg = VATICS_APBC_READL(APBC_DMA_CHN_MONITOR);
	reg &= ~(1<<dma_chn);
	VATICS_APBC_WRITEL(APBC_DMA_CHN_MONITOR, reg);
}

static void vatics_dma_getposition(int chn, dma_addr_t* src, dma_addr_t* dst)
{
	*src = (dma_addr_t)(VATICS_APBC_READL(APBC_DMA_SRC_ADDR(chn)));
	*dst = (dma_addr_t)(VATICS_APBC_READL(APBC_DMA_DES_ADDR(chn)));
}


static void vatics_pcm_start_dma(struct vivo_audio_dev *adev, int direction)
{
	struct vivo_runtime_data *prtd = adev->prtd[direction];
	int dev_num = prtd->dev_num;
	unsigned int dma_size;
	unsigned int index;
	unsigned int offset;
	struct dma_regs_t *desc_addr = prtd->dma_data->desc_addr;
	struct dma_regs_t dma_mmr_init;

	VPL_DEBUG("%s +++\n", __func__);
	dma_size = adev->runtime_dmasize; // transfer one period size data each dma move
	memset(&dma_mmr_init, 0, sizeof(dma_mmr_init));

	if (direction == AUDIO_STREAM_PLAYBACK) {
		dma_mmr_init.dwSrc_Addr = adev->dma_buffer[direction]->dma_addr;
		dma_mmr_init.dwDes_Addr = ssp_tx[dev_num] + I2S_TXDMA;
		dma_mmr_init.dwLLP_Addr = virt_to_phys(desc_addr);
		dma_mmr_init.dwCtrl = 0x50159 |
							(dma_req_sel[1] << 12) |
							((dma_size / 8) << 20);
		for (index = 0; index < (adev->runtime_periods); index++) {
			offset = dma_size * ((index + 1) % adev->runtime_periods);
			desc_addr[index].dwSrc_Addr = dma_mmr_init.dwSrc_Addr + offset;
			desc_addr[index].dwDes_Addr = dma_mmr_init.dwDes_Addr;
			desc_addr[index].dwLLP_Addr = dma_mmr_init.dwLLP_Addr + sizeof(struct dma_regs_t) * ((index + 1) % adev->runtime_periods);
			desc_addr[index].dwCtrl = dma_mmr_init.dwCtrl;
		}
	}
	else {
		dma_mmr_init.dwSrc_Addr = ssp_rx[dev_num] + I2S_RXDMA;
		dma_mmr_init.dwDes_Addr = adev->dma_buffer[direction]->dma_addr;;
		dma_mmr_init.dwLLP_Addr = virt_to_phys(desc_addr);
		dma_mmr_init.dwCtrl = 0x50499 |
							(dma_req_sel[0] << 12) |
							((dma_size / 8) << 20);
		for (index = 0; index < (adev->runtime_periods); index++) {
			offset = dma_size * ((index + 1) % adev->runtime_periods);
			desc_addr[index].dwSrc_Addr = dma_mmr_init.dwSrc_Addr;
			desc_addr[index].dwDes_Addr = dma_mmr_init.dwDes_Addr + offset;
			desc_addr[index].dwLLP_Addr = dma_mmr_init.dwLLP_Addr + sizeof(struct dma_regs_t) * ((index + 1) % adev->runtime_periods);
			desc_addr[index].dwCtrl = dma_mmr_init.dwCtrl;
		}
	}
	/* XXX , please help to review it if the functiont is correct*/
	for (index = 0; index < (adev->runtime_periods); index++) {
		flush_kernel_dcache_page(virt_to_page(&desc_addr[index]));
	}
	vatics_write_dma(&dma_mmr_init, prtd->dma_data->chn);
}

static void vatics_pcm_stop_dma(struct vivo_audio_dev *adev, int direction)
{
	struct vivo_runtime_data *prtd = adev->prtd[direction];
	int dma_chn = prtd->dma_data->chn;
	struct dma_regs_t *llp_addr, *n_llp_addr;

	VPL_DEBUG("%s +++\n", __func__);
	llp_addr = (struct dma_regs_t*)(phys_to_virt(VATICS_APBC_READL(APBC_DMA_LLP(dma_chn))));
	n_llp_addr = (struct dma_regs_t*)(phys_to_virt(llp_addr->dwLLP_Addr));

	// 1. Disable LLP
	llp_addr->dwLLP_Addr = 0x0;
	n_llp_addr->dwLLP_Addr = 0x0;
	flush_cache_all();

	// 2. Wait for DMA Complete
	while (VATICS_APBC_READL(APBC_DMA_LLP(dma_chn)) != 0);
	while (VATICS_APBC_READL(APBC_DMA_CTRL(dma_chn))>>20 != 0);

	// 3. Disable DMA
	vatics_disable_dma(dma_chn);

	// 4. Free LLP
	//if (prtd->dma_data->desc_addr) {
	//	kfree(prtd->dma_data->desc_addr);
	//}
}

static inline void audio_period_elapsed(struct vivo_runtime_data *prtd)
{
	//printk("chn:%d\n", prtd->dma_data->chn);
	if (prtd->dma_data->chn == DMA_CHN_8) { // playback
		blocking_notifier_call_chain(&playback_notifier_list, 0, prtd->adev);
	} else { //capture
		blocking_notifier_call_chain(&capture_notifier_list, 0, prtd->adev);
	}
	return;
}

static irqreturn_t audio_dma_isr(int irq, void *dev_id)
{
	struct vivo_runtime_data *prtd = dev_id;
	unsigned long chn_monitor;
	int index;
	chn_monitor = VATICS_APBC_READL(APBC_DMA_CHN_MONITOR);
	for (index = VATICS_I2S0; index < VATICS_I2S_NUM; index++) {
		if ((1<<(prtd->dma_data->chn)) & chn_monitor) {
			VATICS_APBC_WRITEL(APBC_DMA_CHN_MONITOR, ~(1<<(prtd->dma_data->chn)));
			audio_period_elapsed(prtd);
			return IRQ_HANDLED;
		}
		if ((1<<(prtd->dma_data->chn)) & chn_monitor) {
			VATICS_APBC_WRITEL(APBC_DMA_CHN_MONITOR, ~(1<<(prtd->dma_data->chn)));	
			audio_period_elapsed(prtd);
			return IRQ_HANDLED;
		}
	}
	return IRQ_NONE;
}

static int vatics_dma_irqrequest(struct vivo_audio_dev *adev, int direction)
{
	struct vivo_runtime_data *prtd = adev->prtd[direction];
	struct vivo_pcm_dma_params *dma_data = prtd->dma_data;

	VPL_DEBUG("%s +++\n", __func__);
	spin_lock(&prtd->lock);
	if (dma_data->irq_acks == 0) {
		dma_data->irq = irq_create_mapping(NULL, APBC_IRQ_NUM);
		if(request_irq (dma_data->irq, audio_dma_isr, IRQF_SHARED, "APBC DMA", prtd)) {
			VPL_DEBUG( "irq request fail\n");
			spin_unlock(&prtd->lock);
			return -EBUSY;
		}
		// set Level Trigger for LLP interrupts
		irq_set_irq_type(dma_data->irq, IRQ_TYPE_LEVEL_HIGH);
	}
	dma_data->irq_acks++;
	spin_unlock(&prtd->lock);
	return 0;
}

static void vatics_dma_irqfree(struct vivo_runtime_data *prtd)
{
	struct vivo_pcm_dma_params *dma_data = prtd->dma_data;

	VPL_DEBUG("%s +++\n", __func__);
	spin_lock(&prtd->lock);
	if (dma_data->irq_acks > 1) {
		dma_data->irq_acks--;
	}
	else {
		free_irq(dma_data->irq, prtd);
	}
	spin_unlock(&prtd->lock);
}

int vivo_pcm_trigger(struct vivo_audio_dev *adev, int cmd, int direction)
{
	struct vivo_runtime_data *prtd = adev->prtd[direction];
	int ret = 0;

	VPL_DEBUG("%s +++\n", __func__);
	spin_lock(&prtd->lock);
	switch (cmd) {
	case AUDIO_TRIGGER_START:
		vatics_pcm_start_dma(adev, direction);
		break;

	case AUDIO_TRIGGER_STOP:
		vatics_pcm_stop_dma(adev, direction);
		break;
	default:
		ret = -EINVAL;
		break;
	}
	spin_unlock(&prtd->lock);
	return ret;
}

/* buffer to period offset helper function */
unsigned int addr_to_period_num(struct vivo_dma_buffer *dma_buffer)
{
	return ((long)dma_buffer->app_idx - (long)dma_buffer->area)
		/ dma_buffer->adev->runtime_dmasize;
}

/* period num to address helper function */
void* period_num_to_addr(struct vivo_dma_buffer *dma_buffer, 
					unsigned int num)
{
	return (void*)(dma_buffer->area +
			dma_buffer->adev->runtime_dmasize * num);
}

/* not verified, just one example*/
void* get_pre_hwaddr(struct vivo_dma_buffer *dma_buffer)
{
	struct vivo_audio_dev *adev = dma_buffer->adev;
	unsigned int hw_off = vivo_pcm_pointer(adev, dma_buffer->direction);
	/* if hw_off == 0, means the previous is at boundary.*/
	hw_off = hw_off ? hw_off -1 : dma_buffer->adev->runtime_periods - 1;
	return period_num_to_addr(dma_buffer, hw_off);
}

/* not verified, just one example*/
void* get_next_hwaddr(struct vivo_dma_buffer *dma_buffer)
{
	struct vivo_audio_dev *adev = dma_buffer->adev;
	unsigned int hw_off = vivo_pcm_pointer(adev, dma_buffer->direction);
	/* if hw_off is the last one, we need to reset it to zero */
	hw_off = ((hw_off+1) % dma_buffer->adev->runtime_periods);
	return period_num_to_addr(dma_buffer, hw_off);
}

unsigned int vivo_pcm_pointer(struct vivo_audio_dev *adev, int direction)
{
	struct vivo_runtime_data *prtd = adev->prtd[direction];
	unsigned int offset;
	dma_addr_t count;
	dma_addr_t src, dst;

	spin_lock(&prtd->lock);
	vatics_dma_getposition(prtd->dma_data->chn, &src, &dst);
	if (direction == AUDIO_STREAM_PLAYBACK)
		count = src - adev->dma_buffer[direction]->dma_addr;
	else
		count = dst - adev->dma_buffer[direction]->dma_addr;

	spin_unlock(&prtd->lock);

	offset = count / adev->runtime_dmasize;
	//TODO what's this?
//	if (offset >= runtime->buffer_size)
//		offset = 0;

	return offset;
}


static int vatics_pcm_preallocate_dma_buffer(struct vivo_audio_dev *adev,
						int stream)
{
	struct vivo_dma_buffer *buf = adev->dma_buffer[stream];
	size_t size = 1024 * 1024; // XXX, this value is from the original design

	VPL_DEBUG("%s +++\n", __func__);
	buf->area = dma_alloc_coherent(adev->dev, size, &buf->dma_addr, GFP_KERNEL);
	if (!buf->area)
		return -ENOMEM;

	buf->bytes = size;
	return 0;
}

static void vatics_pcm_free_dma_buffers(struct vivo_audio_dev *adev)
{
	int stream;
	struct vivo_dma_buffer* buf;


	VPL_DEBUG("%s +++\n", __func__);
	for (stream = 0; stream < 2; stream++) {
		buf = adev->dma_buffer[stream];
		if (!buf->area)
			continue;
		dma_free_coherent(adev->dev, buf->bytes, buf->area, buf->dma_addr);
		buf->area = NULL;
	}
}
int vivo_pcm_close(struct vivo_audio_dev *adev)
{
	struct vivo_runtime_data *prtd;
	int i;

	VPL_DEBUG("%s +++\n", __func__);

	for (i = 0 ; i < AUDIO_STREAM_CAPTURE ; i++) {
		prtd = adev->prtd[i];
		vatics_dma_irqfree(prtd);
		kfree(prtd);
	}
	vatics_pcm_free_dma_buffers(adev);
	return 0;
}


static u64 vatics_pcm_dmamask = 0xffffffff;
int vivo_pcm_new(struct vivo_audio_dev *adev, int direction)
{
	struct vivo_runtime_data *prtd;
	struct vivo_pcm_dma_params *dma_data;
	struct vivo_dma_buffer *dma_buffer;
	int ret = 0;

	VPL_DEBUG("%s +++\n", __func__);

	prtd = kzalloc(sizeof(*prtd), GFP_KERNEL);
	if (prtd == NULL)
		return -ENOMEM;
	dma_data = kzalloc(sizeof(*dma_data), GFP_KERNEL);
	if (dma_data == NULL)
		return -ENOMEM;

	dma_buffer = kzalloc(sizeof(*dma_buffer), GFP_KERNEL);
	if (dma_buffer == NULL)
		return -ENOMEM;

	prtd->dma_data = dma_data;
	dma_buffer->adev = adev;
	dma_buffer->direction = direction;
	adev->dma_buffer[direction] = dma_buffer;
	
	vpl_audio_apb_descbuf_capture = \
		(struct dma_regs_t *)kmalloc((PERIOD_BYTES_MAX) * sizeof(struct dma_regs_t), GFP_KERNEL) ;
	vpl_audio_apb_descbuf_playback = \
		(struct dma_regs_t *)kmalloc((PERIOD_BYTES_MAX) * sizeof(struct dma_regs_t), GFP_KERNEL) ;	
	
	if (direction == AUDIO_STREAM_PLAYBACK) {
		dma_data->desc_addr = vpl_audio_apb_descbuf_playback;
		prtd->dma_data->chn = DMA_CHN_8;
	} else { // capture
		dma_data->desc_addr = vpl_audio_apb_descbuf_capture;
		prtd->dma_data->chn = DMA_CHN_6;
	}

	spin_lock_init(&prtd->lock);
	prtd->dev_num = VATICS_I2S0;
	prtd->dma_data = dma_data;
	prtd->adev = adev;
	adev->prtd[direction] = prtd;
	vatics_dma_irqrequest(adev, direction);


	if (!adev->dev->dma_mask)
		adev->dev->dma_mask = &vatics_pcm_dmamask;
	if (!adev->dev->coherent_dma_mask)
		adev->dev->coherent_dma_mask = 0xffffffff;

	if (direction == AUDIO_STREAM_PLAYBACK) {
		ret = vatics_pcm_preallocate_dma_buffer(adev,
					AUDIO_STREAM_PLAYBACK);
		if (ret) {
			VPL_DEBUG("vatics_pcm_new: playback\n");
			return ret;
		}
	} else {// capture
		ret = vatics_pcm_preallocate_dma_buffer(adev, 
					AUDIO_STREAM_CAPTURE);
		if (ret) {
			VPL_DEBUG("vatics_pcm_new: capture\n");
			return ret;
		}
	}


	return 0;
}

