/*
 * Copyright (c) 2003-2013 Broadcom Corporation
 *
 * Copyright (c) 2009-2010 Micron Technology, Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

//nclude <linux/module.h>
//#include <linux/delay.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/partitions.h>
#include <linux/mtd/nand.h>
#include <linux/mtd/nand_ecc.h>
//#include <linux/spi/spi.h>

#include <common.h>
#include <asm/errno.h>
#include <asm/sizes.h>
#include <malloc.h>
#include <spi_flash.h>
#include "../spi/spi_flash_internal.h"
#include "mt29f_spinand.h"

#include <linux/mtd/nand_ecc.h>

#define devm_kzalloc(x,y,z)		malloc(y)
#define GFP_KERNEL
#define pr_info	printf

#define BUFSIZE (10 * 64 * 2048)
#define CACHE_BUF 2112

struct spinand_info *info;
struct spinand_state *state;

static u8 status_reg = 0;

/*
 * OOB area specification layout:  Total 32 available free bytes.
 */
static inline struct spinand_state *mtd_to_state(struct mtd_info *mtd)
{
       // struct nand_chip *chip = (struct nand_chip *)mtd->priv;
        //struct spinand_info *info = info;(struct spinand_info *)chip->priv;
        struct spinand_state *state = (struct spinand_state *)info->priv;

        return state;
}

static int enable_hw_ecc;
static int enable_read_hw_ecc;

struct nand_ecclayout nand_oob_128={0};
unsigned long dma_page_addr = 0 ;

struct nand_ecclayout spinand_mt29f_oob_64 = {
	.eccbytes = 24,
	.eccpos = {
		1, 2, 3, 4, 5, 6,
		17, 18, 19, 20, 21, 22,
		33, 34, 35, 36, 37, 38,
		49, 50, 51, 52, 53, 54, },
	.oobavail = 32,
	.oobfree = { {8, 8}, {24, 8}, {40, 8}, {56, 8}, },
};

struct nand_ecclayout spinand_psu1g_oob_64 = {
	.eccbytes = 28,
	.eccpos = {
		1, 2, 3, 4, 5, 6, 7,
		17, 18, 19, 20, 21, 22, 23,
		33, 34, 35, 36, 37, 38, 39,
		49, 50, 51, 52, 53, 54, 55},
	.oobavail = 32,
	.oobfree = { {8, 8}, {24, 8}, {40, 8}, {56, 8} },
};

struct nand_ecclayout spinand_gd5f_oob_64 = {
	.eccbytes = 16,
	.eccpos = {
		12, 13, 14, 15,
		28, 29, 30, 31,
		44, 45, 46, 47,
		60, 61, 62, 63},
	.oobavail = 32,
	.oobfree = { {4, 8}, {20, 8}, { 36, 8}, {52, 8} },
};

struct nand_ecclayout spinand_wb_oob_64 = {
	.eccbytes = 32,
	.eccpos = {
		8, 9, 10, 11, 12, 13, 14, 15,
		24, 25, 26, 27, 28, 29, 30, 31,
		40, 41, 42, 43, 44, 45, 46, 47,
		56, 57, 58, 59, 60, 61, 62, 63},
	.oobavail = 24,
	.oobfree = { {2, 6}, {18, 6}, { 34, 6}, {50, 6} },
};

struct nand_ecclayout nand_oob_64 = {
        .eccbytes = 24,
        .eccpos = {
                1, 2, 3, 4, 5, 6,
                17, 18, 19, 20, 21, 22,
                33, 34, 35, 36, 37, 38,
                49, 50, 51, 52, 53, 54, },
        .oobavail = 32,
        .oobfree = {
                {.offset = 8,
                        .length = 8},
                {.offset = 24,
                        .length = 8},
                {.offset = 40,
                        .length = 8},
                {.offset = 56,
                        .length = 8},
        }
};

static struct spi_flash *flash ;

/*
 * spinand_cmd - to process a command to send to the SPI Nand
 * Description:
 *    Set up the command buffer to send to the SPI controller.
 *    The command buffer has to initialized to 0.
 */

static int spinand_cmd(struct spi_device *spi, struct spinand_cmd *cmd)
{
        u8 dummy = 0x00;
	u8 *buf = 0;
	u8 *tx=0;
	u32 len = 0;
	u32 i = 0, ret = 0;
	int n_addr = (int)cmd->n_addr;

	len = 1;
        if (cmd->n_addr) {
                len += cmd->n_addr;
        }
        if (cmd->n_dummy) {
                len += cmd->n_dummy;
        }
	tx = malloc(len);
	memset(tx, 0, len);
	
	tx[i++] = cmd->cmd;
	while ( n_addr > 0 ) {
		tx[i] = cmd->addr[i-1];
		i++;
		n_addr--;
	}
	
	if (cmd->n_dummy)
		tx[i++] = 0x00; // dummy
	
	if (cmd->n_rx) {
		ret = spi_flash_read_common(flash, (tx), len, cmd->rx_buf, cmd->n_rx);
	}
	else { 	
		{
			//printf("cmd: %02x %02x%02x%02x, len:%x, tx:%x, ntx:%x\n", (*tx), tx[1], tx[2], tx[3], len, cmd->tx_buf, cmd->n_tx);
			ret = spi_flash_cmd_write(flash->spi, (tx), len, cmd->tx_buf, cmd->n_tx);
		}
	}
	
	free(tx);
	
	return 0;
}

/*
 * spinand_read_id- Read SPI Nand ID
 * Description:
 *    Read ID: read two ID bytes from the SPI Nand device
 */
static int spinand_read_id(struct spi_device *spi_nand, u8 *id)
{
        int retval;
        u8 nand_id[3];
        struct spinand_cmd cmd = {0};

        cmd.cmd = CMD_READ_ID;
	cmd.n_dummy = 1;			// data sheet does have.
        cmd.n_rx = 3;
        cmd.rx_buf = &nand_id[0];

        retval = spinand_cmd(spi_nand, &cmd);
        if (retval < 0) {
                printf("error %d reading id\n", retval);
                return retval;
        }

        id[0] = nand_id[0];
        id[1] = nand_id[1];
		
        return retval;
}

/*
 * spinand_read_status- send command 0xf to the SPI Nand status register
 * Description:
 *    After read, write, or erase, the Nand device is expected to set the
 *    busy status.
 *    This function is to allow reading the status of the command: read,
 *    write, and erase.
 *    Once the status turns to be ready, the other status bits also are
 *    valid status bits.
 */
static int spinand_read_status(struct spi_device *spi_nand, uint8_t *status)
{
        struct spinand_cmd cmd = {0};
        int ret;

        cmd.cmd = CMD_READ_REG;
        cmd.n_addr = 1;
        cmd.addr[0] = REG_STATUS;
        cmd.n_rx = 1;
        cmd.rx_buf = status;

        ret = spinand_cmd(spi_nand, &cmd);
        if (ret < 0)
                printf( "err: %d read status register\n", ret);

        return ret;
}

static int wait_till_ready(struct spi_device *spi_nand)
{
	struct spi_slave *spi = flash->spi;
	unsigned long timebase;
	int ret;
	u8 status;
	u8 cmd[2] = { CMD_READ_REG, REG_STATUS};

	timebase = get_timer(0);
	do {
		ret = spi_flash_cmd_read(spi, cmd, sizeof(cmd), &status, sizeof(status));
		if (ret)
			return -1;

		if (status&STATUS_E_FAIL_MASK&STATUS_P_FAIL_MASK) printf(" status %x \n", status);
		if ((status & STATUS_BUSY) == 0)
			break;

	} while (get_timer(timebase) < SPI_FLASH_PROG_TIMEOUT);

	if ((status & STATUS_BUSY) == 0)
		return 0;

	/* Timed out */
	return -1;
}
/**
 * spinand_get_otp- send command 0xf to read the SPI Nand OTP register
 * Description:
 *   There is one bit( bit 0x10 ) to set or to clear the internal ECC.
 *   Enable chip internal ECC, set the bit to 1
 *   Disable chip internal ECC, clear the bit to 0
 */
static int spinand_get_otp(struct spi_device *spi_nand, u8 *otp)
{
        struct spinand_cmd cmd = {0};
        int retval;

        cmd.cmd = CMD_READ_REG;
        cmd.n_addr = 1;
        cmd.addr[0] = REG_OTP;
        cmd.n_rx = 1;
        cmd.rx_buf = otp;

        retval = spinand_cmd(spi_nand, &cmd);
        if (retval < 0)
                printf( "error %d get otp\n", retval);
        return retval;
}

/**
 * spinand_set_otp- send command 0x1f to write the SPI Nand OTP register
 * Description:
 *   There is one bit( bit 0x10 ) to set or to clear the internal ECC.
 *   Enable chip internal ECC, set the bit to 1
 *   Disable chip internal ECC, clear the bit to 0
 */
static int spinand_set_otp(struct spi_device *spi_nand, u8 *otp)
{
        int retval;
        struct spinand_cmd cmd = {0};

        cmd.cmd = CMD_WRITE_REG,
        cmd.n_addr = 1,
        cmd.addr[0] = REG_OTP,
        cmd.n_tx = 1,
        cmd.tx_buf = otp,

        retval = spinand_cmd(spi_nand, &cmd);
        if (retval < 0)
                printf( "error %d set otp\n", retval);

        return retval;
}


/**
 * spinand_enable_ecc- send command 0x1f to write the SPI Nand OTP register
 * Description:
 *   There is one bit( bit 0x10 ) to set or to clear the internal ECC.
 *   Enable chip internal ECC, set the bit to 1
 *   Disable chip internal ECC, clear the bit to 0
 */
static int spinand_enable_ecc(struct spi_device *spi_nand)
{
        int retval;
        u8 otp = 0;

        retval = spinand_get_otp(spi_nand, &otp);
        if (retval < 0)
                return retval;
	
        if ((otp & OTP_ECC_MASK) == OTP_ECC_MASK) {
                return 0;
        } else {
                otp |= OTP_ECC_MASK;
                retval = spinand_set_otp(spi_nand, &otp);
                if (retval < 0)
                        return retval;
                return spinand_get_otp(spi_nand, &otp);
        }
}

static int spinand_disable_ecc(struct spi_device *spi_nand)
{
        int retval;
        u8 otp = 0;

        retval = spinand_get_otp(spi_nand, &otp);
        if (retval < 0)
                return retval;

        if ((otp & OTP_ECC_MASK) == OTP_ECC_MASK) {
                otp &= ~OTP_ECC_MASK;
                retval = spinand_set_otp(spi_nand, &otp);
                if (retval < 0)
                        return retval;
                return spinand_get_otp(spi_nand, &otp);
        } else
                return 0;
}

/**
 * spinand_write_enable- send command 0x06 to enable write or erase the
 * Nand cells
 * Description:
 *   Before write and erase the Nand cells, the write enable has to be set.
 *   After the write or erase, the write enable bit is automatically
 *   cleared (status register bit 2)
 *   Set the bit 2 of the status register has the same effect
 */
static int spinand_write_enable(struct spi_device *spi_nand)
{
        struct spinand_cmd cmd = {0};

        cmd.cmd = CMD_WR_ENABLE;
        return spinand_cmd(spi_nand, &cmd);
}

static int spinand_read_page_to_cache(struct spi_device *spi_nand, u16 page_id)
{
        struct spinand_cmd cmd = {0};
        u16 row;

        row = page_id;
        cmd.cmd = CMD_READ;
        cmd.n_addr = 3;
        cmd.addr[1] = (u8)((row & 0xff00) >> 8);
        cmd.addr[2] = (u8)(row & 0x00ff);

        return spinand_cmd(spi_nand, &cmd);
}

/*
 * spinand_read_from_cache- send command 0x03 to read out the data from the
 * cache register(2112 bytes max)
 * Description:
 *   The read can specify 1 to 2112 bytes of data read at the corresponding
 *   locations.
 *   No tRd delay.
 */
static int spinand_read_from_cache(struct spi_device *spi_nand, u16 page_id,
                u16 byte_id, u16 len, u8 *rbuf)
{
        struct spinand_cmd cmd = {0};
        u16 column;

        column = byte_id;
        cmd.cmd = CMD_READ_RDM;
        cmd.n_addr = 3;
        cmd.addr[0] = (u8)((column & 0xf00) >> 8);		// bit [11-8]
        //cmd.addr[0] |= (u8)(((page_id >> 6) & 0x1) << 4);
        cmd.addr[1] = (u8)(column & 0x00ff);
        cmd.addr[2] = (u8)(0x00);
        cmd.n_dummy = 0;
        cmd.n_rx = len;
        cmd.rx_buf = rbuf;

        return spinand_cmd(spi_nand, &cmd);
}

/*
 * spinand_read_page-to read a page with:
 * @page_id: the physical page number
 * @offset:  the location from 0 to 2111
 * @len:     number of bytes to read
 * @rbuf:    read buffer to hold @len bytes
 *
 * Description:
 *   The read includes two commands to the Nand: 0x13 and 0x03 commands
 *   Poll to read status to wait for tRD time.
 */
static int spinand_read_page(struct spi_device *spi_nand, u16 page_id,
                u16 offset, u16 len, u8 *rbuf)
{
        int ret;
        u8 status = 0;
		
	status_reg = 0;

#ifdef CONFIG_MTD_SPINAND_ONDIEECC
        if (enable_read_hw_ecc) {
                if (spinand_enable_ecc(spi_nand) < 0)
                        printf( "enable HW ECC failed!");
        }
#endif
        ret = spinand_read_page_to_cache(spi_nand, page_id);
        if (ret < 0)
                return ret;

        if (wait_till_ready(spi_nand))
                printf( "WAIT timedout!!!\n");

        while (1) {
                ret = spinand_read_status(spi_nand, &status);
                if (ret < 0) {
                        printf(
                                        "err %d read status register\n", ret);
                        return ret;
                }

                if ((status & STATUS_OIP_MASK) == STATUS_READY) {
                        if ((status & STATUS_ECC_MASK) == STATUS_ECC_ERROR) {
                                printf( "ecc error, page=%d\n",
                                                page_id);
				status_reg = STATUS_ECC_ERROR;
                                return 0;
                        }
                        break;
                }
        }

        ret = spinand_read_from_cache(spi_nand, page_id, offset, len, rbuf);
        if (ret < 0) {
                printf( "read from cache failed!!\n");
                return ret;
        }

#ifdef CONFIG_MTD_SPINAND_ONDIEECC
        if (enable_read_hw_ecc) {
                ret = spinand_disable_ecc(spi_nand);
                if (ret < 0) {
                        printf( "disable ecc failed!!\n");
                        return ret;
                }
                enable_read_hw_ecc = 0;
        }
#endif
	//printk("ret %d, page_id:%x, offset:%x, len:%x, rbuf:%x\n", ret, page_id, offset, len, rbuf[10]);
        return ret;
}

/*
 * spinand_program_data_to_cache--to write a page to cache with:
 * @byte_id: the location to write to the cache
 * @len:     number of bytes to write
 * @rbuf:    read buffer to hold @len bytes
 *
 * Description:
 *   The write command used here is 0x84--indicating that the cache is
 *   not cleared first.
 *   Since it is writing the data to cache, there is no tPROG time.
 */
static int spinand_program_data_to_cache(struct spi_device *spi_nand,
                u16 page_id, u16 byte_id, u16 len, u8 *wbuf)
{
        struct spinand_cmd cmd = {0};
        u16 column;

        column = byte_id;
        cmd.cmd = CMD_PROG_PAGE_CLRCACHE;
        cmd.n_addr = 2;
        cmd.addr[0] = (u8)((column & 0xff00) >> 8);
        cmd.addr[0] |= (u8)(((page_id >> 6) & 0x1) << 4);
        cmd.addr[1] = (u8)(column & 0x00ff);
        cmd.n_tx = len;
        cmd.tx_buf = wbuf;

        return spinand_cmd(spi_nand, &cmd);
}

/**
 * spinand_program_execute--to write a page from cache to the Nand array with
 * @page_id: the physical page location to write the page.
 *
 * Description:
 *   The write command used here is 0x10--indicating the cache is writing to
 *   the Nand array.
 *   Need to wait for tPROG time to finish the transaction.
 */
static int spinand_program_execute(struct spi_device *spi_nand, u16 page_id)
{
        struct spinand_cmd cmd = {0};
        u16 row;

        row = page_id;
        cmd.cmd = CMD_PROG_PAGE_EXC;
        cmd.n_addr = 3;
        cmd.addr[1] = (u8)((row & 0xff00) >> 8);
        cmd.addr[2] = (u8)(row & 0x00ff);
	
        return spinand_cmd(spi_nand, &cmd);
}

/**
 * spinand_program_page--to write a page with:
 * @page_id: the physical page location to write the page.
 * @offset:  the location from the cache starting from 0 to 2111
 * @len:     the number of bytes to write
 * @wbuf:    the buffer to hold the number of bytes
 *
 * Description:
 *   The commands used here are 0x06, 0x84, and 0x10--indicating that
 *   the write enable is first sent, the write cache command, and the
 *   write execute command.
 *   Poll to wait for the tPROG time to finish the transaction.
 */
static int spinand_program_page(struct spi_device *spi_nand,
                u16 page_id, u16 offset, u16 len, u8 *buf)
{
        int retval;
        u8 status = 0;
        uint8_t *wbuf;
#ifdef CONFIG_MTD_SPINAND_ONDIEECC
	/* Don't use ONDIEECC macro, Oops, always enable now */
        unsigned int i, j;
		
	status_reg = 0;
        enable_read_hw_ecc = 0;
        wbuf = devm_kzalloc(&spi_nand->dev, CACHE_BUF, GFP_KERNEL);
        spinand_read_page(spi_nand, page_id, 0, CACHE_BUF, wbuf);

        for (i = offset, j = 0; i < len; i++, j++)
                wbuf[i] &= buf[j];

        if (enable_hw_ecc) {
                retval = spinand_enable_ecc(spi_nand);
                if (retval < 0) {
                        printf( "enable ecc failed!!\n");
                        return retval;
                }
        }
#else
        wbuf = buf;
#endif
        retval = spinand_write_enable(spi_nand);
        if (retval < 0) {
                printf( "write enable failed!!\n");
                return retval;
        }
        if (wait_till_ready(spi_nand))
                printf( "wait timedout!!!\n");

        retval = spinand_program_data_to_cache(spi_nand, page_id,
                        offset, len, wbuf);
        if (retval < 0)
                return retval;
        retval = spinand_program_execute(spi_nand, page_id);
        if (retval < 0)
                return retval;
        while (1) {
                retval = spinand_read_status(spi_nand, &status);
                if (retval < 0) {
                        printf(
                                        "error %d reading status register\n",
                                        retval);
                        return retval;
                }

                if ((status & STATUS_OIP_MASK) == STATUS_READY) {
                        if ((status & STATUS_P_FAIL_MASK) == STATUS_P_FAIL) {
                                printf(
                                        "program error, page %d\n", page_id);
				status_reg = STATUS_P_FAIL;
                                return -1;
                        } else
                                break;
                }
        }
#ifdef CONFIG_MTD_SPINAND_ONDIEECC
        if (enable_hw_ecc) {
                retval = spinand_disable_ecc(spi_nand);
                if (retval < 0) {
                        printf( "disable ecc failed!!\n");
                        return retval;
                }
                enable_hw_ecc = 0;
        }
#endif

        return 0;
}

/**
 * spinand_erase_block_erase--to erase a page with:
 * @block_id: the physical block location to erase.
 *
 * Description:
 *   The command used here is 0xd8--indicating an erase command to erase
 *   one block--64 pages
 *   Need to wait for tERS.
 */
static int spinand_erase_block_erase(struct spi_device *spi_nand, u16 block_id)
{
        struct spinand_cmd cmd = {0};
        u16 row;

        row = block_id;
        cmd.cmd = CMD_ERASE_BLK;
        cmd.n_addr = 3;
        cmd.addr[1] = (u8)((row & 0xff00) >> 8);
        cmd.addr[2] = (u8)(row & 0x00ff);

        return spinand_cmd(spi_nand, &cmd);
}

/**
 * spinand_erase_block--to erase a page with:
 * @block_id: the physical block location to erase.
 *
 * Description:
 *   The commands used here are 0x06 and 0xd8--indicating an erase
 *   command to erase one block--64 pages
 *   It will first to enable the write enable bit (0x06 command),
 *   and then send the 0xd8 erase command
 *   Poll to wait for the tERS time to complete the tranaction.
 */
static int spinand_erase_block(struct spi_device *spi_nand, u16 block_id)
{
        int retval;
        u8 status = 0;

	status_reg = 0;
        retval = spinand_write_enable(spi_nand);
        if (wait_till_ready(spi_nand))
                printf( "wait timedout!!!\n");


        retval = spinand_erase_block_erase(spi_nand, block_id);
        while (1) {
                retval = spinand_read_status(spi_nand, &status);
                if (retval < 0) {
                        printf(
                                        "error %d reading status register\n",
                                        (int) retval);
                        return retval;
                }

                if ((status & STATUS_OIP_MASK) == STATUS_READY) {
                        if ((status & STATUS_E_FAIL_MASK) == STATUS_E_FAIL) {
                                printf(
                                        "erase error, block %d\n", block_id);
				status_reg = STATUS_E_FAIL;
                                return -1;
                        } else
                                break;
                }
        }
        return 0;
}


void spinand_write_page_hwecc(struct mtd_info *mtd,
                struct nand_chip *chip, const uint8_t *buf)//, int oob_required)
{
        const uint8_t *p = buf;
        int eccsize = chip->ecc.size;
        int eccsteps = chip->ecc.steps;

        enable_hw_ecc = 1;
        chip->write_buf(mtd, p, eccsize * eccsteps +64);
       // return 0;
}
 
static int spinand_read_page_hwecc(struct mtd_info *mtd, struct nand_chip *chip,
                uint8_t *buf)//, int oob_required, int page)
{
        u8 retval, status;
        uint8_t *p = buf;
        int eccsize = chip->ecc.size;
        int eccsteps = chip->ecc.steps;
        //struct spinand_info *info = (struct spinand_info *)chip->priv;

        enable_read_hw_ecc = 1;

        chip->read_buf(mtd, p, eccsize * eccsteps);
        if (1)//oob_required)
                chip->read_buf(mtd, chip->oob_poi, mtd->oobsize);

        while (1) {
                retval = spinand_read_status(info->spi, &status);
                if ((status & STATUS_OIP_MASK) == STATUS_READY) {
                        if ((status & STATUS_ECC_MASK) == STATUS_ECC_ERROR) {
                                pr_info("spinand: ECC error\n");
                                mtd->ecc_stats.failed++;
                        } else if ((status & STATUS_ECC_MASK) ==
                                        STATUS_ECC_1BIT_CORRECTED)
                                mtd->ecc_stats.corrected++;
                        break;
                }
        }
        return 0;

}


static void spinand_select_chip(struct mtd_info *mtd, int dev)
{
}

static uint8_t spinand_read_byte(struct mtd_info *mtd)
{
        struct spinand_state *state = mtd_to_state(mtd);
        u8 data;

        data = state->buf[state->buf_ptr];
        state->buf_ptr++;
        return data;
}


static int spinand_wait(struct mtd_info *mtd, struct nand_chip *chip)
{
	u8 ret = status_reg;
	status_reg = 0;
	return ret?NAND_STATUS_FAIL:0;
}

static void spinand_write_buf(struct mtd_info *mtd, const uint8_t *buf, int len)
{
        struct spinand_state *state = mtd_to_state(mtd);

        memcpy(state->buf + state->buf_ptr, buf, len);
        state->buf_ptr += len;
}

static void spinand_read_buf(struct mtd_info *mtd, uint8_t *buf, int len)
{
        struct spinand_state *state = mtd_to_state(mtd);

        memcpy(buf, state->buf + state->buf_ptr, len);
        state->buf_ptr += len;
}

static int spinand_lock_block(struct spi_device *spi_nand, u8 lock);
/*
 * spinand_reset- send RESET command "0xff" to the Nand device.
 */
static void spinand_reset(struct spi_device *spi_nand)
{
        struct spinand_cmd cmd = {0};
		
        cmd.cmd = CMD_RESET;

        if (spinand_cmd(spi_nand, &cmd) < 0)
                pr_info("spinand reset failed!\n");

        /* elapse 1ms before issuing any other command */
        udelay(1000);

        if (wait_till_ready(spi_nand))
                printf( "wait timedout!\n");
	spinand_lock_block(info->spi, 0);	
}

static void spinand_cmdfunc(struct mtd_info *mtd, unsigned int command,
                int column, int page)
{
        //struct nand_chip *chip = (struct nand_chip *)mtd->priv;
       //struct spinand_info *info = info;//(struct spinand_info *)chip->priv;
        struct spinand_state *state = (struct spinand_state *)info->priv;
	int i = 0;
	
	//printf("spinand_cmd:%x, page:%x, col:%x\n", command, page, column);
	
        switch (command) {
        /*
         * READ0 - read in first  0x800 bytes
         */
        case NAND_CMD_PAGEREAD_BUFMODE:
        case NAND_CMD_READ1:
        case NAND_CMD_READ0:
                state->buf_ptr = 0;
                spinand_read_page(info->spi, page, 0x0, 0x840, state->buf);
	#if 0
		for ( i = 0x800; i < 0x810; i++) {
			printf("%02x ", state->buf[i]);
			if (  !((i+1)%16) ) printf("\n");
		}
	#endif
                break;
        /* READOOB reads only the OOB because no ECC is performed. */
        case NAND_CMD_READOOB:
                state->buf_ptr = 0;
                spinand_read_page(info->spi, page, 0x800, 0x40, state->buf);
	#if 0
		for ( i = 0; i < 0x10; i++) {
			printf("%02x ", state->buf[i]);
			if (  !((i+1)%16) ) printf("\n");
		}
	#endif
                break;
        case NAND_CMD_RNDOUT:
                state->buf_ptr = column;
                break;
        case NAND_CMD_READID:
                state->buf_ptr = 0;
                spinand_read_id(info->spi, (u8 *)state->buf);
                break;/*
        case NAND_CMD_PARAM:
                state->buf_ptr = 0;
                break;*/
        /* ERASE1 stores the block and page address */
        case NAND_CMD_ERASE1:
                spinand_erase_block(info->spi, page);
                break;
        /* ERASE2 uses the block and page address from ERASE1 */
        case NAND_CMD_ERASE2:
                break;
        /* SEQIN sets up the addr buffer and all registers except the length */
        case NAND_CMD_SEQIN:
                state->col = column;
                state->row = page;
                state->buf_ptr = 0;
                break;
        /* PAGEPROG reuses all of the setup from SEQIN and adds the length */
        case NAND_CMD_PAGEPROG:
                spinand_program_page(info->spi, state->row, state->col,
                                state->buf_ptr, state->buf);
                break;
        case NAND_CMD_STATUS:
                spinand_get_otp(info->spi, state->buf);
                if (!(state->buf[0] & 0x80))
                        state->buf[0] = 0x80;
                state->buf_ptr = 0;
                break;
        /* RESET command */
        case NAND_CMD_RESET:
                if (wait_till_ready(info->spi))
                        printf("WAIT timedout!!!\n");
                /* a minimum of 250us must elapse before issuing RESET cmd*/
                udelay(250);
                spinand_reset(info->spi);
                break;
        default:
                printf("Unknown CMD: 0x%x\n", command);
        }
}

/**
 * spinand_lock_block- send write register 0x1f command to the Nand device
 *
 * Description:
 *    After power up, all the Nand blocks are locked.  This function allows
 *    one to unlock the blocks, and so it can be written or erased.
 */
static int spinand_lock_block(struct spi_device *spi_nand, u8 lock)
{
        struct spinand_cmd cmd = {0};
        int ret;
        u8 otp = 0;
	
        ret = spinand_get_otp(spi_nand, &otp);

        cmd.cmd = CMD_WRITE_REG;
        cmd.n_addr = 1;
        cmd.addr[0] = REG_BLOCK_LOCK;
        cmd.n_tx = 1;
        cmd.tx_buf = &lock;

        ret = spinand_cmd(spi_nand, &cmd);
        if (ret < 0)
                printf( "error %d lock block\n", ret);

        return ret;
}
#if 0
/*
 * spinand_probe - [spinand Interface]
 * @spi_nand: registered device driver.
 *
 * Description:
 *   To set up the device driver parameters to make the device available.
 */
static int spinand_probe(struct spi_device *spi_nand)
{ 
        struct mtd_info *mtd;
        struct nand_chip *chip;
        struct spinand_info *info;
        struct spinand_state *state;
        struct mtd_part_parser_data ppdata;

        info  = devm_kzalloc(&spi_nand->dev, sizeof(struct spinand_info),
                        GFP_KERNEL);
        if (!info)
                return -ENOMEM;

        info->spi = spi_nand;

        spinand_lock_block(spi_nand, BL_ALL_UNLOCKED);
	
        state = devm_kzalloc(&spi_nand->dev, sizeof(struct spinand_state),
                        GFP_KERNEL);
        if (!state)
                return -ENOMEM;

        info->priv      = state;
        state->buf_ptr  = 0;
        state->buf      = devm_kzalloc(&spi_nand->dev, BUFSIZE, GFP_KERNEL);
        if (!state->buf)
                return -ENOMEM;

        chip = devm_kzalloc(&spi_nand->dev, sizeof(struct nand_chip),
                        GFP_KERNEL);
        if (!chip)
                return -ENOMEM;

#ifdef CONFIG_MTD_SPINAND_ONDIEECC
        chip->ecc.mode  = NAND_ECC_HW;
        chip->ecc.size  = 0x200;
        chip->ecc.bytes = 0x4;//6;
        chip->ecc.steps = 0x4;

        chip->ecc.strength = 1;
        chip->ecc.total = chip->ecc.steps * chip->ecc.bytes;
        chip->ecc.layout = &spinand_oob_64;
        chip->ecc.read_page = spinand_read_page_hwecc;
        chip->ecc.write_page = spinand_write_page_hwecc;
#else
        chip->ecc.mode  = NAND_ECC_SOFT;
        if (spinand_disable_ecc(spi_nand) < 0)
                pr_info("%s: disable ecc failed!\n", __func__);
#endif

        chip->priv      = info;
        chip->read_buf  = spinand_read_buf;
        chip->write_buf = spinand_write_buf;
        chip->read_byte = spinand_read_byte;
        chip->cmdfunc   = spinand_cmdfunc;
        chip->waitfunc  = spinand_wait;
        chip->options   |= NAND_CACHEPRG;
        chip->select_chip = spinand_select_chip;

        mtd = devm_kzalloc(&spi_nand->dev, sizeof(struct mtd_info), GFP_KERNEL);
        if (!mtd)
                return -ENOMEM;

        dev_set_drvdata(&spi_nand->dev, mtd);

        mtd->priv = chip;
        mtd->name = dev_name(&spi_nand->dev);
//      mtd->owner = THIS_MODULE;
        mtd->oobsize = 64;

        if (nand_scan(mtd, 1))
                return -ENXIO;

        ppdata.of_node = spi_nand->dev.of_node;
        return mtd_device_parse_register(mtd, NULL, &ppdata, NULL, 0);
}

/*
 * spinand_remove: Remove the device driver
 * @spi: the spi device.
 *
 * Description:
 *   To remove the device driver parameters and free up allocated memories.
 */
static int spinand_remove(struct spi_device *spi)
{
        mtd_device_unregister(dev_get_drvdata(&spi->dev));

        return 0;
}

static const struct of_device_id spinand_dt[] = {
        { .compatible = "spinand,mt29f", },
        {}
};

/*
 * Device name structure description
 */
static struct spi_driver spinand_driver = {
        .driver = {
                .name           = "mt29f",
                .bus            = &spi_bus_type,
//              .owner          = THIS_MODULE,
                .of_match_table = spinand_dt,
        },
        .probe          = spinand_probe,
        .remove         = spinand_remove,
};
#endif

static int nand_dev_ready( struct mtd_info *mtd ) 
{
	struct spi_slave *spi = flash->spi;
	unsigned long timebase;
	int ret;
	u8 status;
	u8 cmd[2] = { 0x0f, 0xC0};

	timebase = get_timer(0);
	do {
		ret = spi_flash_cmd_read(spi, cmd, sizeof(cmd), &status, sizeof(status));
		if (ret)
			return -1;

		debug(" status %x \n", status);
		if ((status & STATUS_OIP_MASK) == 0)
			break;

	} while (get_timer(timebase) < SPI_FLASH_PROG_TIMEOUT);

	if ((status & STATUS_OIP_MASK) == 0)
		return 0;

	/* Timed out */
	return -1;
}

int nand_get_desc_page_number(void* tmp, struct mtd_info* mtd) 
{
	struct nand_bbt_descr* desc = tmp ;
	struct nand_chip* chip = mtd->priv ;
	int bits = desc->options & NAND_BBT_NRBITS_MSK;
	int bbt_length = ((chip->chipsize / mtd->erasesize) * bits) >> 3;

	bbt_length = (bbt_length + (mtd->writesize - 1)) & ~(mtd->writesize - 1);//make bbt_length page-aligned

	return (bbt_length / mtd->writesize) + 1 ;
}

static int vpl_nand_scan(struct nand_chip *chip, struct vpl_nand_chip_private *chip_priv)
{
	//int err = 0 ;    
	
#ifdef CONFIG_MTD_PARTITIONS
	struct mtd_partition *mtd_parts = 0;
	int mtd_parts_nr = 0;
#endif

	flash = spi_flash_probe(CONFIG_SPI_FLASH_BUS, 0, CONFIG_SF_DEFAULT_SPEED, CONFIG_DEFAULT_SPI_MODE);

        info  = malloc(sizeof(struct spinand_info));
        if (!info)
                return -ENOMEM;
		
	state = malloc( sizeof(struct spinand_state));
	if (!state)
			return -ENOMEM;
	
        info->priv      = state;
	state->buf_ptr	= 0;
	state->buf		= malloc( BUFSIZE);
	if (!state->buf)
			return -ENOMEM;

	//struct nand_chip init 
	chip->options |= NAND_NO_SUBPAGE_WRITE | NAND_USE_FLASH_BBT;//[jam test]NAND_BBT_SCANEMPTY for testing...
	chip->IO_ADDR_W = NULL;//(void  __iomem *)VPL_NFC_MMR_DATA ;
	chip->IO_ADDR_R = NULL;//chip->IO_ADDR_W ;

	chip->select_chip = spinand_select_chip;
	chip->cmdfunc = spinand_cmdfunc;
	chip->dev_ready = nand_dev_ready;
	chip->cmd_ctrl = NULL;//nand_hwcontrol ;//This is used to issue command and address cycles to the chip (CLE/ALE)
	chip->write_page = NULL ;//nand_write_page ;
	chip->read_byte = spinand_read_byte;
	chip->read_buf = spinand_read_buf;
	chip->write_buf = spinand_write_buf;
    	chip->waitfunc = spinand_wait ;///nand_program_erase_waitfunc ;

	chip->ecc.read_page = spinand_read_page_hwecc ;//nand_ecc_read_page_hwecc
	chip->ecc.write_page = spinand_write_page_hwecc ;//nand_ecc_write_page_hwecc
	chip->ecc.layout = &spinand_mt29f_oob_64 ;
	chip->ecc.calculate = NULL ;//nand_calculate ;
	chip->ecc.correct = NULL ;//nand_correct ;
	chip->ecc.hwctl = NULL ;//nand_hwctl ;
	chip->ecc.mode = NAND_ECC_HW ;
	chip->ecc.size = 0x40;//0;
	
	//struct vpl_nand_chip_private init
	chip_priv->nf_chip_timing = 0xffffffff ;//default timing
	chip_priv->get_desc_page_number = nand_get_desc_page_number ;    

	//Since DMA cannot work on some chips(SAMSUNG), we use CPU mode instead.
    //chip_priv->dma_mode = 1 ;
    chip_priv->dma_mode = 0 ;

    return 0 ;
}

int board_nand_init(struct nand_chip *chip)
{
	struct vpl_nand_chip_private *chip_priv ;
	int err = 0 ;

	chip_priv = (struct vpl_nand_chip_private *)malloc(sizeof(struct vpl_nand_chip_private)) ;
	if (chip_priv == NULL) {
		printf( "[ERR]No memory for chip_priv of nand\n" );
		err = -ENOMEM;
		goto probe_exit_error;
	}

	//main structure assignment
	chip->priv = chip_priv ;

	//we init fields of struct nand_chip
    if( vpl_nand_scan(chip, chip_priv) != 0 ) {
		printf( "[ERR]init nandchip failed!!\n" );
		err = -1;
		goto probe_exit_error;
	}

	return 0 ;

probe_exit_error :
	if (err == 0)
		err = -EINVAL;
	return err;
}


