#include <common.h>
#include <malloc.h>
#include <net.h>
#include <netdev.h>

#include <asm/arch/platform.h>
#include "gmac.h"

#if !defined(CONFIG_MII)
# error "GMAC requires MII -- missing CONFIG_MII"
#endif

#if !defined(CONFIG_PHYLIB)
# error "GMAC requires PHYLIB -- missing CONFIG_PHYLIB"
#endif

//-1 : no quick-test
// 0 : force to use 10/100Mbps
// 1 : force to use 1Gbps
int gmac_quick_testspeed = -1;

static struct dw_eqos_priv gmac_priv;
u8 MAC_Addr[6] = {0x00, 0xab, 0xcd, 0xab, 0xcd, 0xef};

static void __inline__ GMACWriteMacReg(struct dw_eqos_priv *priv, u32 Reg, u32 Data)
{
	v_outl(priv->macBase+Reg, Data);
}

static u32 __inline__ GMACReadMacReg(struct dw_eqos_priv *priv, u32 Reg )
{
	u32 data = v_inl(priv->macBase+Reg);
	return data;
}

static void __inline__ GMACSetMacReg(struct dw_eqos_priv *priv, u32 Reg, u32 Data )
{
	u32 addr = priv->macBase+Reg;
	u32 data = v_inl(addr);
	data |= Data;
	v_outl(addr, data);
}

static void __inline__ GMACClearMacReg(struct dw_eqos_priv *priv, u32 Reg, u32 Data )
{
	u32 addr = priv->macBase+Reg;
	u32 data = v_inl(addr);
	data &= ~Data;
	v_outl(addr, data);
}

static u32 __inline__ GMACReadDmaReg(struct dw_eqos_priv *priv, u32 Reg)
{
	u32 data = v_inl(priv->dmaBase+Reg);
	return data;
}

static void __inline__ GMACWriteDmaReg(struct dw_eqos_priv *priv, u32 Reg, u32 Data)
{
	v_outl(priv->dmaBase+Reg, Data);
}

static void __inline__ GMACSetDmaReg(struct dw_eqos_priv *priv, u32 Reg, u32 Data)
{
	u32 addr = priv->dmaBase+Reg;
	u32 data = v_inl(addr);
	data |= Data;
	v_outl(addr, data);
}

static void __inline__ GMACClearDmaReg(struct dw_eqos_priv *priv, u32 Reg, u32 Data )
{
	u32 addr = priv->dmaBase+Reg;
	u32 data = v_inl(addr);
	data &= ~Data;
	v_outl(addr, data);
}

int GMACMacInit(struct dw_eqos_priv *priv)
{
	struct phy_device *phydev = priv->phydev;
	u32 conf;

	conf = GMAC_CONTROL_IPC | GMAC_CONTROL_BE | GMAC_CONTROL_DO;

	if (phydev->speed != SPEED_1000)
		conf |= GMAC_CONTROL_PS;

	if (phydev->speed == SPEED_100)
		conf |= GMAC_CONTROL_FES;

	if (phydev->duplex)
		conf |= GMAC_CONTROL_DM;

	conf |= GMAC_CONTROL_TE | GMAC_CONTROL_RE;

	GMACWriteMacReg(priv, GmacPacketFilter, (1<<5));
	GMACWriteMacReg(priv, GmacConfig, conf); /* set init values of config registers with MII port */

	/* Don't need to set filter */
	/* Disable broadcast only in loopback quick test */

	return 0;
}

static void tx_descs_init(struct dw_eqos_priv *priv)
{
	u32 idx;
	struct tx_desc *desc_p;

	priv->txCount = TX_DESC_COUNT;
	priv->tx = (struct tx_desc *)priv->tx_desc_base;

	for (idx = 0; idx < TX_DESC_COUNT; idx++) {
		desc_p = priv->tx + idx;
		desc_p->buffer1_addr = (u32)(priv->tx_buff + (idx * DMA_BUFFER_SIZE));
		desc_p->buffer2_addr = 0;
		desc_p->length = 0;
		desc_p->status = 0;
	}

	GMACWriteDmaReg(priv, DmaCH0TxDescListAddr, priv->tx_desc_base);
	GMACWriteDmaReg(priv, DmaCH0TxDescRingLength, TX_DESC_COUNT - 1);
	GMACWriteDmaReg(priv, DmaCH0TxDescTailAddr,  0);
	priv->txNext = 0;
	priv->txBusy = 0;
}

void fill_rx(struct dw_eqos_priv *priv)
{
	struct rx_desc *desc_p;
	priv->rxBusy = priv->rxNext;
	desc_p = priv->rx + priv->rxNext;
	priv->rxNext = (priv->rxNext + 1) % RX_DESC_COUNT;
	/* DMA will do write back, must refill each time */
	desc_p->buffer1_addr = (u32)(priv->rx_buff + priv->rxBusy * DMA_BUFFER_SIZE);
	desc_p->status = DESC_RXCTL_OWN | DESC_RXCTL_BUF1V;
	GMACWriteDmaReg(priv, DmaCH0RxDescTailAddr, (u32)(priv->rx + RX_DESC_COUNT));
}

static void rx_descs_init(struct dw_eqos_priv *priv)
{
	u32 idx;
	struct rx_desc *desc_p;

	priv->rxCount = RX_DESC_COUNT;
	priv->rx = (struct rx_desc *)priv->rx_desc_base;

	for (idx = 0; idx < RX_DESC_COUNT; idx++) {
		desc_p = priv->rx + idx;
		desc_p->buffer1_addr = (u32)(priv->rx_buff + idx * DMA_BUFFER_SIZE); /*will be over write once DMA done*/
		desc_p->buffer2_addr = 0;
		desc_p->status = 0;
	}

	/* Setup the Base and Length of the Rx Descriptor Ring */
	GMACWriteDmaReg(priv, DmaCH0RxDescListAddr, priv->rx_desc_base);
	GMACWriteDmaReg(priv, DmaCH0RxDescRingLength, RX_DESC_COUNT -1);
	/* Setup the HW Rx Tail Descriptor Pointers */
	GMACWriteDmaReg(priv, DmaCH0RxDescTailAddr, (u32)(priv->rx + RX_DESC_COUNT));
	priv->rxNext = 0;
	priv->rxBusy = 0;

	fill_rx(priv);
}

static void desc_init(struct dw_eqos_priv *priv)
{
	tx_descs_init(priv);
	rx_descs_init(priv);
}

static void gmac_mtl_init(struct dw_eqos_priv *priv)
{
	GMACWriteMacReg(priv, GmacMtlTxQ0Operation, MTL_TXQ_TSF);
	GMACWriteMacReg(priv, GmacMtlRxQ0Operation, MTL_RXQ_FUP | MTL_RXQ_RSF);
}

static int gmac_dma_init(struct dw_eqos_priv *priv)
{
	GMACWriteDmaReg(priv, DmaCH0TxControl, DMA_CH_BURST_16);
	GMACWriteDmaReg(priv, DmaCH0RxControl, DMA_CH_BURST_16);
	return 0;
}

/*
 * Assign Interface. Overwrite by steps
 * 1. enviroment variable phy_mode= rgmii, mii, rmii
 * 2. HW setting (V1 only)
 * 3. Chip limitation (M325, M330, M330s are support MII only)
 */
static phy_interface_t gmac_get_phy_mode(void)
{
	/* Default GMII */
	phy_interface_t	interface = PHY_INTERFACE_MODE_GMII;
	phy_interface_t mode;
	char *phy_mode_env;

	/* Enviroment variable: backward compatible for rgmii=0/1 */
	phy_mode_env = getenv("rgmii");
	if(phy_mode_env != NULL)
		if (simple_strtoul (phy_mode_env, NULL, 10))
			interface = PHY_INTERFACE_MODE_RGMII;

	/* Enviroment variable: New environment variable "phy_mode" */
	phy_mode_env = getenv("phy_mode");
	for (mode=PHY_INTERFACE_MODE_MII; mode< PHY_INTERFACE_MODE_NONE; mode++) {
		if (strcmp(phy_mode_env, phy_interface_strings[mode]) == 0) {
			interface = mode;
			break;
		}
	}

	return interface;
}

static void gmac_phy_init(struct dw_eqos_priv *priv)
{
	struct eth_device *fake_dev;
	struct phy_device *phydev;
	phy_interface_t interface = priv->interface;

	fake_dev = (struct eth_device *)malloc(sizeof(struct eth_device)); /*TODO: free*/
	sprintf(fake_dev->name, "GMAC");

	/* phy_connect implicate phy_reset */
	phydev = phy_connect(priv->mdio_bus, priv->phyAddr, fake_dev, priv->interface);

	if (phydev) {
		/*
		 * By default GMII is used, but also work for MII PHY,
		 * auto fix environment variable if PHY driver told it doesn't support 1000baseT.
		 */
		if (interface == PHY_INTERFACE_MODE_GMII) {
			if (!((phydev->drv->features & SUPPORTED_1000baseT_Half) ||
			      (phydev->drv->features & SUPPORTED_1000baseT_Full))) {
				interface = priv->interface = phydev->interface = PHY_INTERFACE_MODE_MII;
				setenv("phy_mode", (char *)phy_interface_strings[PHY_INTERFACE_MODE_MII]);
				saveenv();
			}
		}
		priv->phydev = phydev;
		/*
		 * Stop Advertising 1000BASE Capability if interface is not GMII
		 * Useful in case PHY supports GMII but used in MII only,
		 * and Hardware miss to limit poweron advertising capability.
		 * Can fix it by setting phy_mode = "mii"
		 */
		if ((interface == PHY_INTERFACE_MODE_MII) ||
				(interface == PHY_INTERFACE_MODE_RMII))
			phydev->advertising &= ~(SUPPORTED_1000baseT_Half |
					SUPPORTED_1000baseT_Full);

		/* phy_config calls driver config, generally include change advert and auto-neg */
		phy_config(phydev);
	}
}

static void gmac_mac_mode_init(struct dw_eqos_priv *priv)
{
	/*
	 * To set GMAC work at correct interface mode.
	 * The following settings must be confirmed when reset MAC DMA Engine.
	 *
	 * 1. Set SYSC_GMAC_CTRL 0x30
	 *   [25:24] GMAC_EXT_CLK_FREQ_SEL:
	 *                     2'b00 125MHz; 2'b01 25MHz; 2'b10 2.5MHz; 2'b11 50MHz(RMII)
	 *   [21:20] GMAC_RMII_CLK_PHASE_SEL:
	 *                     2'b00 Normal; 2'b01 90; 2'b10 180; 2'b11 270
	 *   [18:16] GMAC_MODE_SEL:
	 *                     3'b000 GMII/MII; 3'b001 RGMII; 3'b100 RMII
	 *      [12] GMAC_TX_CLK_SRC:
	 *                     1'b0 Internal MAC->PHY; 1'b1 External MAC<-PHY
	 *       [8] GMAC_RX_CLK_SRC:
	 *                     1'b0 External MAC<-PHY (default); 1'b1 Internal MAC->PHY (used in RMII)
	 *       [4] GMAC_REDUCED_MODE_EN: (TX_CLK PAD to PHY)
	 *	               1'b0: 180 degree, 1'b1: 90 degree
	 *   [01:00] GMAC_INT_CLK_FREQ_SEL:
	 *                     2'b00 125MHz; 2'b01 25MHz; 2'b10 2.5MHz;
	 *
	 *          | GMAC_RX_CLK_DIR | GMAC_TX_CLK_DIR | GMAC_MODE_SEL | GMAC_EXT_CLK_FREQ_SEL
	 * --------------------------------------------------------------------------------------
	 * MII      |    1'b1         |    1'b0         |  3'b000       |
	 * RGMII    |    1'b1         |    1'b0         |  3'b001       |
	 * RMII     |    1'b1         |    1'b0         |  3'b100       |	 2'b11 (50MHz for PHY)
	 *
	 * 2. Reset GMAC DMA in driver.
	 *
	 * After Reset DMA Engine, the Tx direction and Speed are free to change
	 * accroding to PHY's speed.
	 */

	u32 val;

	switch (priv->interface) {
		case PHY_INTERFACE_MODE_MII:
			val = 0x00000000;
			break;
		case PHY_INTERFACE_MODE_RGMII:
			val = 0x00011010;
			break;
		case PHY_INTERFACE_MODE_RMII:
			v_outl(EVM_SYSC_BASE + 0x94, v_inl(EVM_SYSC_BASE + 0x94) | (1<<13));
			val = 0x03041100;
			break;
		default:
			break;
	}
	v_outl(EVM_SYSC_BASE + 0x30, val);
}

static void gmac_fix_mac_speed(struct phy_device *phydev)
{
	/*
	 *   [25:24] GMAC_EXT_CLK_FREQ_SEL:
	 *                     2'b00 125MHz; 2'b01 25MHz; 2'b10 2.5MHz; 2'b11 50MHz(RMII)
	 *   [21:20] GMAC_RMII_CLK_PHASE_SEL:
	 *                     2'b00 Normal; 2'b01 90; 2'b10 180; 2'b11 270
	 *   [18:16] GMAC_MODE_SEL:
	 *                     3'b000 GMII/MII; 3'b001 RGMII; 3'b100 RMII
	 *      [12] GMAC_TX_CLK_SRC:
	 *                     1'b0 External MAC<-PHY; 1'b1 Internal MAC->PHY;
	 *       [8] GMAC_RX_CLK_SRC:
	 *                     1'b0 External MAC<-PHY (default); 1'b1 Internal MAC->PHY (used in RMII)
	 *       [4] GMAC_REDUCED_MODE_EN: (TX_CLK PAD to PHY)
	 *	               1'b0: 180 degree, 1'b1: 90 degree
	 *   [01:00] GMAC_INT_CLK_FREQ_SEL:
	 *                     2'b00 125MHz; 2'b01 25MHz; 2'b10 2.5MHz;
	 */

	u32 val = v_inl(EVM_SYSC_BASE + 0x30);
	switch (phydev->interface) {
		case PHY_INTERFACE_MODE_MII:
			/* speed = 100 or 10, Tx Clk and Rx Clk are external */
			val &= ~(1<<12 | 1<<8);
			/* reg = 0x1<<16 | 0x1<<12 | 0x0 << 8; */
			break;
		case PHY_INTERFACE_MODE_RGMII:
			switch (phydev->speed) {
				case SPEED_1000:
					val = 0x0<<24 | 0x1<<16 | 0x1 <<12 | 0x0<<8 | 0x1 << 4 | 0x0<<0;
					break;
				case SPEED_100:
					val = 0x1<<24 | 0x1<<16 | 0x1 <<12 | 0x0<<8 | 0x1 << 4 | 0x1<<0;
					break;
				case SPEED_10:
					val = 0x2<<24 | 0x1<<16 | 0x1 <<12 | 0x0<<8 | 0x1 << 4 | 0x2<<0;
					break;
			}
			break;
		case PHY_INTERFACE_MODE_RMII:
			switch (phydev->speed) {
				case SPEED_100:
					val = 0x3<<24 | 0x4<<16 | 0x1<<12 | 0x1<<8 | 0x1<<0 ;
					break;
				case SPEED_10:
					val = 0x3<<24 | 0x4<<16 | 0x1<<12 | 0x1<<8 | 0x2<<0 ;
					break;
			}
			break;
		default:
			break;
	}
	v_outl(EVM_SYSC_BASE + 0x30, val);
}

static void gmac_reset(struct dw_eqos_priv *priv)
{
	int rs_timeout = 0;

	GMACWriteDmaReg(priv, DmaBusMode, DMA_MODE_SFT_RESET);
	while(GMACReadDmaReg(priv, DmaBusMode) & DMA_MODE_SFT_RESET) {
		if (rs_timeout > 10000) {
			printf("[GMAC] Fatal Error: Reset GMAC timeout, not complete over 1sec\n");
			break;
		}
		rs_timeout++;
		udelay(100);
	};
/*
	switch (GMACReadDmaReg(priv, GmacHWFeature0) >> 28) {
		case 0x0:
			printf("[GMAC] GMII interface mode\n");
			break;
		case 0x1:
			printf("[GMAC] RGMII interface mode\n");
			break;
		case 0x4:
			printf("[GMAC] RMII interface mode\n");
			break;
		default:
			printf("[GMAC] Fatal Error, unsupported interface\n");
			break;
	}
*/
}

void gmac_set_ethaddr(struct dw_eqos_priv *priv, u8 *Addr)
{
	unsigned long data ;

	data = (Addr[5]<<8) | Addr[4];                            /* set our MAC address */
	GMACWriteMacReg(priv, GmacAddr0High, data);
	data = (Addr[3]<<24) | (Addr[2]<<16) | (Addr[1]<<8) | Addr[0];
	GMACWriteMacReg(priv, GmacAddr0Low, data);
}

void GMACInit(int quiet)
{
	struct dw_eqos_priv *priv = &gmac_priv;
	struct phy_device *phydev;

	gmac_mac_mode_init(priv);

	/*
	 * Before reset, must do gmac_mac_mode_init.
	 * It is essential that all PHY inputs clocks are present for software reset completion
	 */
	gmac_reset(priv);

	phydev = priv->phydev;

	/* Print status */
	if(!quiet) printf("  PHY TYPE     : %s(%s)\n", phydev->drv->name, phy_interface_strings[priv->interface]);

	if (gmac_quick_testspeed == -1) { /* Normal auto-nego */
		if (NET_FIX_SPEED == 0) { /* audo nego */
			phydev->autoneg = AUTONEG_ENABLE;
			phydev->speed = 0;
			phydev->duplex = -1;
		} else { /* fixed speed */
			phydev->autoneg = AUTONEG_DISABLE;
			phydev->speed = SPEED_100;
			phydev->duplex = DUPLEX_FULL;
		}
		phydev->link = 1;
		phy_config(phydev);

		/* Get link status */
		if (!quiet) printf("  Link Detect  : ");
		if (NET_FIX_SPEED == 0) { /* audo nego */
			phy_startup(phydev);
		} else {
			/* Do a fake read, or fix speed config will not be applied */
			phy_read(phydev, MDIO_DEVAD_NONE, MII_BMSR);
			printf("%s Fixed speed mode\n",
				phydev->dev->name);
		}
	}

	if(!quiet) printf("  Link Status  : %s\n", (phydev->link)?"Link Up":"Link Down");

	if(phydev->link == 0) {
		if(!quiet) printf("                 - Cannot detect link connected signal in the limited time...\n");
		if(!quiet) printf("                 - Assume it works in 100Mbps + Full-duplex.\n");
		if (gmac_quick_testspeed == -1) {
			phydev->autoneg = AUTONEG_DISABLE;
			phydev->speed = SPEED_100;
			phydev->duplex = DUPLEX_FULL;
			phy_config(phydev);
			/* Do a fake read, or fix speed config will not be applied */
			phy_read(phydev, MDIO_DEVAD_NONE, MII_BMSR);
		}
	} else {
		if(!quiet) printf("%s", (phydev->duplex == DUPLEX_FULL) ? "               : Full Duplex\n" : "               : Half-Duplex\n");
		switch(phydev->speed) {
			case SPEED_1000:
				if(!quiet) printf("  Link Speed   : 1000Mbps\n");
				break;
			case SPEED_100:
				if(!quiet) printf("  Link Speed   : 100Mbps\n");
				break;
			case SPEED_10:
				if(!quiet) printf("  Link Speed   : 10Mbps\n");
				break;
			default:
				if(!quiet) printf("  Link Speed   : Unknown\n");
				break;
		}
	}

	/* Copy the MAC addr into the HW  */
	if(!quiet) printf("  MAC Address  : %02x %02x %02x %02x %02x %02x\n", MAC_Addr[0], MAC_Addr[1], MAC_Addr[2], MAC_Addr[3], MAC_Addr[4], MAC_Addr[5]);
	gmac_set_ethaddr(priv, MAC_Addr);

	/* Initial Tx/Rx Descriptor and Buffer */
	desc_init(priv);

	/* Initial MTL FIFO Queue */
	gmac_mtl_init(priv);

	/* Initial DMA */
	gmac_dma_init(priv);

	gmac_fix_mac_speed(priv->phydev);

	//-------------------------------
	// setup_net_GMAC
	//-------------------------------
	if(GMACMacInit(priv) != 0)
	{
		printf("GMAC - cannot init MAC\n");
		return;
	}

}

int eth_init(bd_t* bd)
{
	struct dw_eqos_priv *priv = &gmac_priv;

	//Start DMA
	GMACSetDmaReg(priv, DmaCH0TxControl, DMA_CH_TX_ST);
	GMACSetDmaReg(priv, DmaCH0RxControl, DMA_CH_RX_SR);
	return 0;
}

void eth_halt(void)
{
	struct dw_eqos_priv *priv = &gmac_priv;

	GMACClearDmaReg(priv, DmaCH0TxControl, DMA_CH_TX_ST);
	GMACClearDmaReg(priv, DmaCH0RxControl, DMA_CH_RX_SR);
}

int eth_rx(void)
{
	struct dw_eqos_priv *priv = &gmac_priv;
	struct rx_desc *rx_desc_p = priv->rx + priv->rxBusy;
	int len = 0;

	if(!(rx_desc_p->status & DESC_RXCTL_OWN)) {
		len = rx_desc_p->status & DESC_RXSTS_FRMLENMSK;
		/* TODO:check valid status GMACDmaRxValid ?*/
		NetReceive((u8 *)priv->rx_buff + priv->rxBusy * DMA_BUFFER_SIZE, len-4 ); /*?*/
		fill_rx(priv);
	}

	return 0;
}

int eth_send(volatile void *packet, int length)
{
	struct dw_eqos_priv *priv = &gmac_priv;
	int desc = priv->txNext;
	struct tx_desc *tx_desc_p = priv->tx + desc;

	if (++desc >= TX_DESC_COUNT)
		desc = 0;
	priv->txNext = desc;

	memcpy((void *)tx_desc_p->buffer1_addr, (const void *)packet, length);

	tx_desc_p->length = ((length << DESC_TX_R_BUF1_LEN_SHIFT) & DESC_TX_R_BUF1_LEN_MASK);
	tx_desc_p->status = DESC_TXCTL_OWN | DESC_TXCTL_FD | DESC_TXCTL_LD;


	GMACWriteDmaReg(priv, DmaCH0TxDescTailAddr, (u32)priv->rx);

	while(tx_desc_p->status & DESC_TXCTL_OWN)
	{
		udelay(5);
	}

	return length;
}

struct dw_eqos_priv* gmac_returnTPrivate(void)
{
	return &gmac_priv ;
}

static void pll2_adjust(void)
{
#define PLLC_2_CTRL 0x14
#define PLLC_2_DIV  0x18
	u32 pllc_base = VPL_PLLC_MMR_BASE;
	u32 plldiv = 0x007c0200;
	int limit = 10;

	v_outl(pllc_base+PLLC_2_DIV, plldiv);
	v_outl(pllc_base+PLLC_2_CTRL, 0x1);
	while(limit--) {
		if ((v_inl(pllc_base+PLLC_2_CTRL) & 0x31) == 0x30)
			break;
		udelay(100);
	}
	if (limit < 0)
		printf("gmac: Can't lock stable PLL clock output for a limited time\n");
}
int gmac_eth_initialize(int quiet)
{
	struct dw_eqos_priv *priv = &gmac_priv;

	pll2_adjust();

	eth_getenv_enetaddr("ethaddr", MAC_Addr);

	/* register phy libs */
	phy_init();

	/* Get GMAC and PHY interface mode */
	priv->interface = gmac_get_phy_mode();

	/* DMA Descriptor allocate */
	priv->desc = (struct tx_desc *)DMA_BUFFER_BASE;
	priv->tx_desc_base = DMA_BUFFER_BASE;
	priv->rx_desc_base = DMA_BUFFER_BASE + (sizeof(struct tx_desc)*TX_DESC_COUNT);
	priv->tx_buff = (char *)(DMA_BUFFER_BASE + (sizeof(struct tx_desc)*DESC_COUNT));
	priv->rx_buff = (char *)(DMA_BUFFER_BASE + (sizeof(struct tx_desc)*DESC_COUNT) + (DMA_BUFFER_SIZE*TX_DESC_COUNT));

	memset((void *)priv->desc, 0, (sizeof(struct tx_desc)*DESC_COUNT));

	priv->macBase = GMAC_MMR_BASE;
	priv->dmaBase = GMAC_MMR_BASE + 0x1000;

	priv->phyAddr = NET_PHY_ADDRESS;

	/* Initialize the MDIO bus */
	gmac_mdio_init(priv);

	/* Connect PHY and config auto-nego */
	gmac_phy_init(priv);

	GMACInit(quiet);

	return 0;
}
