#include <common.h>
#include <spi_flash.h>
#include <asm/io.h>
#include <asm/arch/platform.h>
#include <i2c.h>
#include "oled.h"



DECLARE_GLOBAL_DATA_PTR;

#ifdef OLED_SSD1306
const struct oledinfo oled_info = {
		.addr = 0x3C,
		.reg_len = 1,
		.width = 128,
		.height = 64,
};

char oled_init_command[] =
{
	0xD5, 0x80, 0xA8, 0x27, 0xD3, 0x00, 0xB0, 0x00,
	0x10, 0x40, 0x20, 0x02, 0xA0, 0xC0, 0xDA, 0x12,
	0x81, 0xAF, 0xD9, 0x25, 0xDB, 0x20, 0xA4, 0xA6
};

char oled_backlight_on_command[] =
{
	0x8D, 0x14, 0xAF
};

char oled_backlight_off_command[] =
{
	0xAE, 0x8D, 0x10
};
#endif

int oled_i2c_write_register(struct oledinfo* oled_info, uchar cmd)
{
	i2c_write(oled_info->addr, 0x80, oled_info->reg_len, &cmd, 1);
	return 0;
}

int oled_write_dram(struct oledinfo* oled_info, char page, int column, char* data, u16 len)
{
	uchar buf[512];
	int i, j;

	oled_i2c_write_register(oled_info, 0xB0 + page);            //set page address
	oled_i2c_write_register(oled_info, 0x10 | (column >> 4));   //set column address
	oled_i2c_write_register(oled_info, column & 0x0F);

	j = 0;

	for(i=0; i < len; i++)
	{
		buf[j] = data[i];
		j++;

		if(((column + i) >= (oled_info->width - 1)) || (i >= (len -1)))
		{
			break;
		}

	}


	if(i2c_write(oled_info->addr, 0x40, oled_info->reg_len, buf, j) != j)
	{
//		printf("oled write dram data fail\n");
//		return -1;
	}
	return 0;
}

int oled_backlight(struct oledinfo* oled_info, int enable)
{
	int i;

	if(enable)
	{
		for(i = 0; i < sizeof(oled_backlight_on_command); i++)
		{
			oled_i2c_write_register(oled_info, oled_backlight_on_command[i]);
		}
	}
	else
	{
		for(i = 0; i < sizeof(oled_backlight_off_command); i++)
		{
			oled_i2c_write_register(oled_info, oled_backlight_off_command[i]);
		}
	}

	return 0;
}

int oled_display(struct oledinfo* oled_info, char* buf)
{
    int i, offset = 0;

    for(i = 0; i < (oled_info->height / 8); i++)
    {
        if (oled_write_dram(oled_info, 0 + i, 0, buf + offset, oled_info->width) == -1)
        {
            return -1;
        }
        offset += oled_info->width;
    }

    return 0;
}

void oled_init(struct oledinfo* oled_info)
{
	int i;

	u32 reg_value = 0;

	reg_value = v_inl(VPL_GPIOC_2_MMR_BASE + GPIOC_DATA_OUT);
	reg_value |= (OLED_SCL | OLED_SDA);
	v_outl(VPL_GPIOC_2_MMR_BASE + GPIOC_DATA_OUT, reg_value);

	reg_value = v_inl(VPL_GPIOC_2_MMR_BASE + GPIOC_PIN_DIR);
	reg_value |= (OLED_SCL | OLED_SDA);
	v_outl(VPL_GPIOC_2_MMR_BASE + GPIOC_PIN_DIR, reg_value);

	SOFT_I2C_SCL_PIN = OLED_SCL;
	SOFT_I2C_SDA_PIN = OLED_SDA;
//    i2c_init (CONFIG_SYS_I2C_SPEED, CONFIG_SYS_I2C_SLAVE);

	for(i = 0; i < sizeof(oled_init_command); i++)
	{
		oled_i2c_write_register(oled_info, oled_init_command[i]);
	}
}

void oled_hw_reset(void)
{
	u32 reg_value = 0;

	if (OLED_RESET < 0)
	{
		return;
	}

	reg_value = v_inl(VPL_GPIOC_1_MMR_BASE + GPIOC_DATA_OUT);
    reg_value |= (OLED_RESET);
	v_outl(VPL_GPIOC_1_MMR_BASE + GPIOC_DATA_OUT, reg_value);

	reg_value = v_inl(VPL_GPIOC_1_MMR_BASE + GPIOC_PIN_DIR);
	reg_value |= (OLED_RESET);
	v_outl(VPL_GPIOC_1_MMR_BASE + GPIOC_PIN_DIR, reg_value);

	reg_value = v_inl(VPL_GPIOC_1_MMR_BASE + GPIOC_DATA_OUT);
    reg_value &= ~(OLED_RESET);
	v_outl(VPL_GPIOC_1_MMR_BASE + GPIOC_DATA_OUT, reg_value);

	udelay(10000);

	reg_value = v_inl(VPL_GPIOC_1_MMR_BASE + GPIOC_DATA_OUT);
    reg_value |= (OLED_RESET);
	v_outl(VPL_GPIOC_1_MMR_BASE + GPIOC_DATA_OUT, reg_value);


}

#define SPLASH_STORAGE_NAND		0x01
#define SPLASH_STORAGE_SF		0x02

#ifdef CONFIG_SPI_FLASH
static struct spi_flash *sf;

extern struct spi_flash *get_spi_flash(void);
static int raw_sf_read(u32 bmp_load_addr, int offset, size_t read_size)
{
	if (!sf) {
		sf = get_spi_flash();
		if (!sf)
			return -ENODEV;
	}
	debug("bmp_load_addr %x %x %d\n", bmp_load_addr, offset, read_size);
	return spi_flash_read(sf, offset, read_size, (void *)bmp_load_addr);
}
#else
static int raw_sf_read(u32 bmp_load_addr, int offset, size_t read_size)
{
	debug("%s: sf support not available\n", __func__);
	return -ENOSYS;
}
#endif

#ifdef CONFIG_CMD_NAND
static int raw_nand_read(u32 bmp_load_addr, int offset, size_t read_size)
{
	return nand_read_skip_bad(&nand_info[nand_curr_device], offset,
				  &read_size, (u_char *)bmp_load_addr);
}
#else
static int raw_nand_read(u32 bmp_load_addr, int offset, size_t read_size)
{
	debug("%s: nand support not available\n", __func__);
	return -ENOSYS;
}
#endif

static int splash_storage_read(struct splash_location *location,
			       u32 bmp_load_addr, size_t read_size)
{
	u32 offset;

	if (!location)
		return -EINVAL;

	offset = location->offset;
	switch (location->storage) {
	case SPLASH_STORAGE_NAND:
		return raw_nand_read(bmp_load_addr, offset, read_size);
	case SPLASH_STORAGE_SF:
		return raw_sf_read(bmp_load_addr, offset, read_size);
	default:
		debug("Unknown splash location\n");
	}

	return -EINVAL;
}

static struct splash_location oled_splash_locations[] = {
	{
		.name = "sf",
		.storage = SPLASH_STORAGE_SF,
		.offset = 0x01FC0000,
	},
    {
        .name = "n2k",
        .storage = SPLASH_STORAGE_NAND,
        .offset = 0x07F40000,
    },
};

int splash_screen_prepare(u32 addr, size_t size)
{
#if (PLATFORM_SPIFLASH_NUM != 0)
    return splash_storage_read(&oled_splash_locations[0], addr, size);
#elif (PLATFORM_NANDFLASH_NUM != 0 || PLATFORM_SPI_NANDFLASH_NUM != 0)
	return splash_storage_read(&oled_splash_locations[1], addr, size);
#endif
}

int drv_oled_init(void)
{
	char *addr1, *addr2;
	char *s;
	u32 fbmem_size = oled_info.width * oled_info.height;

    int i, j;
    size_t write_size = 0;
    int buffer_pos = 0;
    int buffer_pos_x;
    int buffer_pos_y;
    int after_byte;
    int after_bit;
    char value;
    u32 reg_value = 0;

	s = getenv("splashimage");
	if (s != NULL) {
		addr1 = simple_strtoul(s, NULL, 16);
	} else {
#if 0
		addr1 = (u32)memalign(8, fbmem_size);
#else
		//addr = 0x10000000 - 0x40000;
		addr1 = gd->bd->bi_dram[0].size - ((fbmem_size + 4095) & (~4095));
		addr2 = addr1 - ((fbmem_size + 4095) & (~4095));
#endif
	}

	if (addr1 == 0) {
		debug("Failed to alloc FB memory\n");
		return -1;
	}

	splash_screen_prepare(addr1, fbmem_size);


	buffer_pos_x = buffer_pos % oled_info.width;
	buffer_pos_y = buffer_pos / oled_info.width;


	printf("before\n");

//	for(i = 0; i < (oled_info.width * oled_info.height); i++)
//	{
//		if((i % 8) == 0)
//		{
//			printf("\n");
//		}
//		printf("%2.2X, ", addr1[i]);
//	}

	for(i = buffer_pos_y; i < oled_info.height; i++)
	{
		after_bit = i % 8;
		after_byte = (i / 8) * oled_info.width;
		for(j = buffer_pos_x; j < oled_info.width; j++)
		{
			value = 1 << after_bit;

			if(addr1[write_size] == 0xFF)
			{
				addr2[j + after_byte] &= (~value);
			}
			else
			{
				addr2[j + after_byte] |= value;
			}

			buffer_pos++;
			write_size ++;

			if(buffer_pos >= (oled_info.width * oled_info.height))
			{
				break;
			}
		}

		buffer_pos_x = 0;
	}

//	printf("after\n");
//	for(i = 0; i < (oled_info.width * oled_info.height); i++)
//	{
//		if((i % 8) == 0)
//		{
//			printf("\n");
//		}
//		printf("%2.2X, ", addr2[i]);
//	}


	reg_value = v_inl(VPL_SYSC_MMR_BASE + 0xAC);
	reg_value |= (OLED_SCL | OLED_SDA);
	v_outl(VPL_SYSC_MMR_BASE + 0xAC, reg_value);

	oled_hw_reset();
	oled_init(&oled_info);
	oled_display(&oled_info, addr2);
	oled_backlight(&oled_info, 1);

	reg_value = v_inl(VPL_SYSC_MMR_BASE + 0xAC);
	reg_value &= ~(OLED_SCL | OLED_SDA);
	v_outl(VPL_SYSC_MMR_BASE + 0xAC, reg_value);

	return 0;
}

