
#include <common.h>
#include <spi_flash.h>
#include <spi.h>
#include <asm/io.h>
#include <asm/arch/platform.h>
#include "vpl_voc.h"
#include <nand.h>

#define SYSC_RST_CTRL 24

#define VOC_CLK_EN_BIT_MASK (1<<18)
#define VOC_RST_N_BIT_MASK (1<<16)

extern void udelay(unsigned long usec);
extern int voc_compute_pll_and_div(u32 pclk_target, u32 *f_estim, u32 *fb_div_p, u32 *ref_div_p, u32 *post_div_p, u32 *voc_div_p);

DECLARE_GLOBAL_DATA_PTR;
struct vpl_voc_info *voc = (struct vpl_voc_info *)VPL_VOC_MMR_BASE;

#if (defined PANEL_NT39016)
const struct videomode modes = {
	.pixelclock = 27000000,
	.hactive = 320,
	.hfront_porch = 228,
	.hback_porch = 23,
	.hsync_len = 1,
	.vactive = 240,
	.vfront_porch = 10,
	.vback_porch = 12,
	.vsync_len = 1,
	.hsync_act_pol = VOC_SYNC_ACTIVE_LOW,
	.vsync_act_pol = VOC_SYNC_ACTIVE_LOW,
	.de_act_pol = VOC_DE_ACTIVE_HIGH,
	.pclk_pol = POSITIVE_EDGE_ALIGNED,
};
#elif (defined PANEL_ILI9341V)
const struct videomode modes = {
    .pixelclock = 18500000,
    .hactive = 240,
    .hfront_porch = 4, /* unit is PCLOCK, *3 = DOTCLK in SRGB(8-bit RGB) mode, 12DOTCLK */
    .hback_porch = 7, /* 21 DOTCLK */
    .hsync_len = 4, /* 12 DOTCLK*/
    .vactive = 320,
    .vfront_porch = 4,
    .vback_porch = 2,
    .vsync_len = 2,
    .hsync_act_pol = VOC_SYNC_ACTIVE_LOW,
    .vsync_act_pol = VOC_SYNC_ACTIVE_LOW,
    .de_act_pol = VOC_DE_ACTIVE_HIGH,
    .pclk_pol = NEGATIVE_EDGE_ALIGNED,
};
#elif (defined PANEL_ILI9342C)
const struct videomode modes = {
	.pixelclock = 18500000,
	.hactive = 320,
	.hfront_porch = 4, /* unit is PCLOCK, *3 = DOTCLK in SRGB(8-bit RGB) mode, 12DOTCLK */
	.hback_porch = 7, /* 21 DOTCLK */
	.hsync_len = 4, /* 12 DOTCLK*/
	.vactive = 240,
	.vfront_porch = 4,
	.vback_porch = 2,
	.vsync_len = 2,
	.hsync_act_pol = VOC_SYNC_ACTIVE_LOW,
	.vsync_act_pol = VOC_SYNC_ACTIVE_LOW,
	.de_act_pol = VOC_DE_ACTIVE_HIGH,
	.pclk_pol = NEGATIVE_EDGE_ALIGNED,
};
#elif (defined PANEL_ST7789)
const struct videomode modes = {
    .pixelclock = 18500000,
    .hactive = 240,
    .hfront_porch = 4, /* unit is PCLOCK, *3 = DOTCLK in SRGB(8-bit RGB) mode, 12DOTCLK */
    .hback_porch = 7, /* 21 DOTCLK */
    .hsync_len = 4, /* 12 DOTCLK*/
    .vactive = 320,
    .vfront_porch = 4,
    .vback_porch = 2,
    .vsync_len = 2,
    .hsync_act_pol = VOC_SYNC_ACTIVE_LOW,
    .vsync_act_pol = VOC_SYNC_ACTIVE_LOW,
    .de_act_pol = VOC_DE_ACTIVE_HIGH,
    .pclk_pol = NEGATIVE_EDGE_ALIGNED,
};

struct regtabl {
    u16 cmd;
    u8 value[15];
    u32 num;
}st7789s_init_tbl[] = {
    //Display and color format setting
    {0x36, {0x08}, 1},
    {0x3a, {0x66}, 1}, //262k: 0X66; 65K: 0X55
    //ST7789S Frame rate setting
    {0xb2, {0x08, 0x08, 0x00, 0x22, 0x22}, 5},
    {0xb7, {0x35}, 1},
    //ST7789S Power setting
    {0xbb, {0x27}, 1},
    {0xc0, {0x2c}, 1},
    {0xc2, {0x01}, 1},
    {0xc3, {0x0b}, 1},
    {0xc4, {0x20}, 1},
    {0xc6, {0x0f}, 1},
    {0xd0, {0xa4, 0xa1}, 2},
    //ST7789S gamma setting
    {0xe0, {0xd0, 0x00, 0x02, 0x07, 0x07, 0x19, 0x2e, 0x54, 0x41, 0x2d, 0x17, 0x18, 0x14, 0x18}, 14},
    {0xe1, {0xd0, 0x00, 0x02, 0x07, 0x04, 0x24, 0x2c, 0x44, 0x42, 0x1c, 0x1a, 0x17, 0x15, 0x18}, 14},
};
#elif (defined PANEL_ST7793)
const struct videomode modes = {
    .pixelclock = 11520000,
    .hactive = 400,
    .hfront_porch = 4, /* unit is PCLOCK, *3 = DOTCLK in SRGB(8-bit RGB) mode, 12DOTCLK */
    .hback_porch = 2, /* 21 DOTCLK */
    .hsync_len = 2, /* 12 DOTCLK*/
    .vactive = 240,
    .vfront_porch = 10,
    .vback_porch = 20,
    .vsync_len = 10,
    .hsync_act_pol = VOC_SYNC_ACTIVE_LOW,
    .vsync_act_pol = VOC_SYNC_ACTIVE_LOW,
    .de_act_pol = VOC_DE_ACTIVE_HIGH,
    .pclk_pol = NEGATIVE_EDGE_ALIGNED,
};

struct regtabl {
    u16 cmd;
    u8 value[15];
    u32 num;
}st7793_init_tbl[] = {
        //Display and color format setting
        {0x0001, {0x01, 0x00}, 2},
        {0x0003, {0xD0, 0x28}, 2},

        {0x0008, {0x04, 0x04}, 2},
        {0x0090, {0x80, 0x00}, 2},

        {0x0400, {0x62, 0x00}, 2},
        {0x0401, {0x00, 0x01}, 2},

        {0x0102, {0x01, 0xB0}, 2},

        {0x0200, {0x00, 0x00}, 2},
        {0x0201, {0x00, 0x00}, 2},

        {0x00FF, {0x00, 0x01}, 2},
        {0x0724, {0x00, 0x1F}, 2},

        {0x0210, {0x00, 0x00}, 2},
        {0x0211, {0x00, 0xEF}, 2},
        {0x0212, {0x00, 0x00}, 2},
        {0x0213, {0x01, 0x8F}, 2},

        {0x0007, {0x01, 0x00}, 2},
};
#elif (defined PANEL_ST7789V)
const struct videomode modes = {
		.pixelclock = 15000000,
		.hactive = 240,
		.hfront_porch = 8, /* unit is PCLOCK, *3 = DOTCLK in SRGB(8-bit RGB) mode, 24DOTCLK */
		.hback_porch = 8,  /* 24 DOTCLK */
		.hsync_len = 6,    /* 18 DOTCLK*/
		.vactive = 240,
		.vfront_porch = 8,
		.vback_porch = 8,
		.vsync_len = 4,
		.hsync_act_pol = VOC_SYNC_ACTIVE_LOW,
		.vsync_act_pol = VOC_SYNC_ACTIVE_LOW,
		.de_act_pol = VOC_DE_ACTIVE_HIGH,
		.pclk_pol = NEGATIVE_EDGE_ALIGNED,
};
#elif (defined PANEL_RM68172_4INCH)
const struct videomode modes = {
	.pixelclock = 27000000,
	.hactive = 480,
	.hfront_porch = 24,
	.hback_porch = 16,
	.hsync_len = 8,
	.vactive = 800,
	.vfront_porch = 18,
	.vback_porch = 14,
	.vsync_len = 2,
	.hsync_act_pol = VOC_SYNC_ACTIVE_LOW,
	.vsync_act_pol = VOC_SYNC_ACTIVE_LOW,
	.de_act_pol = VOC_DE_ACTIVE_LOW,
	.pclk_pol = NEGATIVE_EDGE_ALIGNED,
};
#elif (defined PANEL_RM68172_5INCH)
const struct videomode modes = {
	.pixelclock = 27000000,
	.hactive = 480,
	.hfront_porch = 24,
	.hback_porch = 16,
	.hsync_len = 8,
	.vactive = 854,
	.vfront_porch = 18,
	.vback_porch = 14,
	.vsync_len = 2,
	.hsync_act_pol = VOC_SYNC_ACTIVE_LOW,
	.vsync_act_pol = VOC_SYNC_ACTIVE_LOW,
	.de_act_pol = VOC_DE_ACTIVE_LOW,
	.pclk_pol = NEGATIVE_EDGE_ALIGNED,
};
#endif

#if (defined PANEL_NT39016) || (defined PANEL_ILI9341V) || (defined PANEL_ILI9342C) || (defined PANEL_ST7789V)
#define SPI_CS				(1 << 9)	//GPIOC2[9]
#define SPI_CLK				(1 << 10)	//GPIOC2[10]
#define SPI_DATA			(1 << 13)	//GPIOC2[13]
#define BACK_LIGHT_EN		(1 << 14)	//GPIOC2[14]

#define DATA_OUT			1
#define DATA_IN				0

#define GPIOC_DATA_OUT		(VPL_GPIOC_2_MMR_BASE + 0x04)
#define GPIOC_DATA_IN		(VPL_GPIOC_2_MMR_BASE + 0x08)
#define GPIOC_PIN_DIR		(VPL_GPIOC_2_MMR_BASE + 0x0C)
#define GPIOC_SET_DATA		(VPL_GPIOC_2_MMR_BASE + 0x14)
#define GPIOC_CLEAR_DATA	(VPL_GPIOC_2_MMR_BASE + 0x18)
#define GPIOC_PULL_EN		(VPL_GPIOC_2_MMR_BASE + 0x1C)
#define GPIOC_PULL_TYPE		(VPL_GPIOC_2_MMR_BASE + 0x20)

#define SYSC_PAD_EN_CTRL_2	(VPL_SYSC_MMR_BASE + 0xA4)
#define SYSC_PAD_EN_CTRL_4	(VPL_SYSC_MMR_BASE + 0xAC)
#define SYSC_PAD_EN_CTRL_5	(VPL_SYSC_MMR_BASE + 0xB0)
#define SYSC_PAD_EN_CTRL_6	(VPL_SYSC_MMR_BASE + 0xB4)

#define SET_IO_LEVEL(pin, level) \
{ \
	if (level) { \
		readl(GPIOC_SET_DATA) |= pin; \
	} \
	else { \
		readl(GPIOC_CLEAR_DATA) |= pin; \
	} \
}\

#define SET_DATA_DIR(pin, dir) \
{ \
	u32 value; \
	value = readl(GPIOC_PIN_DIR); \
	if (dir == DATA_OUT) { \
		value |= pin; \
	} \
	else { \
		value &= ~pin; \
	} \
	writel(value, GPIOC_PIN_DIR); \
}\

#define GET_DATA_LEVEL(pin)		(readl(GPIOC_DATA_IN) & pin)

static void gpio_init(void)
{
	u32 value;

#if 1 // SPI GPIO pinmux init
	// set pad enable
	value = readl(SYSC_PAD_EN_CTRL_4);
	value |= SPI_CS | SPI_CLK | SPI_DATA | BACK_LIGHT_EN;
	writel(value, SYSC_PAD_EN_CTRL_4);

	// disable UARTC_1, GPIOC2_13 and 14
	value = readl(SYSC_PAD_EN_CTRL_5);
	value &= ~(1 << 29 | 1 << 30);
	writel(value, SYSC_PAD_EN_CTRL_5);

	// disable UARTC_3, GPIOC2_9 and 10
	value = readl(SYSC_PAD_EN_CTRL_6);
	value &= ~(1 << 1 | 1 << 2);
	writel(value, SYSC_PAD_EN_CTRL_6);
#endif

	// set output direct
	value = readl(GPIOC_PIN_DIR);
	value |= SPI_CS | SPI_CLK | SPI_DATA | BACK_LIGHT_EN;
	writel(value, GPIOC_PIN_DIR);

	// set level
	SET_IO_LEVEL(SPI_CS, 1);
	SET_IO_LEVEL(SPI_CLK, 1);
	SET_IO_LEVEL(SPI_DATA, 0);
	SET_IO_LEVEL(BACK_LIGHT_EN, 1);

	// set pull high
	value = readl(GPIOC_PULL_TYPE);
	value |= SPI_CS | SPI_CLK | SPI_DATA | BACK_LIGHT_EN;
	writel(value, GPIOC_PULL_TYPE);

	// set pull enable
	value = readl(GPIOC_PULL_EN);
	value |= SPI_CS | SPI_CLK | SPI_DATA | BACK_LIGHT_EN;
	writel(value, GPIOC_PULL_EN);

	debug("0x%x = 0x%x\n", SYSC_PAD_EN_CTRL_4, readl(SYSC_PAD_EN_CTRL_4));
	debug("0x%x = 0x%x\n", SYSC_PAD_EN_CTRL_5, readl(SYSC_PAD_EN_CTRL_5));
	debug("0x%x = 0x%x\n", SYSC_PAD_EN_CTRL_6, readl(SYSC_PAD_EN_CTRL_6));

	debug("0x%x = 0x%x\n", GPIOC_PIN_DIR, readl(GPIOC_PIN_DIR));
	debug("0x%x = 0x%x\n", GPIOC_DATA_OUT, readl(GPIOC_DATA_OUT));
	debug("0x%x = 0x%x\n", GPIOC_DATA_SET, readl(GPIOC_DATA_SET));
	debug("0x%x = 0x%x\n", GPIOC_DATA_CLEAR, readl(GPIOC_DATA_CLEAR));
	debug("0x%x = 0x%x\n", GPIOC_PULL_TYPE, readl(GPIOC_PULL_TYPE));
	debug("0x%x = 0x%x\n", GPIOC_PULL_EN, readl(GPIOC_PULL_EN));
}
#elif (defined PANEL_RM68172_4INCH) || (defined PANEL_RM68172_5INCH)
#define SPI_CS				(1 << 11)
#define SPI_CLK				(1 << 13)
#define SPI_DATA			(1 << 9)
#define BACK_LIGHT_EN		(1 << 14)
#define PANEL_RESET			(1 << 14)

#define DATA_OUT			1
#define DATA_IN				0

#define GPIOC_DATA_OUT		(VPL_GPIOC_1_MMR_BASE + 0x04)
#define GPIOC_DATA_IN		(VPL_GPIOC_1_MMR_BASE + 0x08)
#define GPIOC_PIN_DIR		(VPL_GPIOC_1_MMR_BASE + 0x0C)
#define GPIOC_SET_DATA		(VPL_GPIOC_1_MMR_BASE + 0x14)
#define GPIOC_CLEAR_DATA	(VPL_GPIOC_1_MMR_BASE + 0x18)
#define GPIOC_PULL_EN		(VPL_GPIOC_1_MMR_BASE + 0x1C)
#define GPIOC_PULL_TYPE		(VPL_GPIOC_1_MMR_BASE + 0x20)

#define SYSC_PAD_EN_CTRL_2	(VPL_SYSC_MMR_BASE + 0xA4)
#define SYSC_PAD_EN_CTRL_3	(VPL_SYSC_MMR_BASE + 0xA8)
#define SYSC_PAD_EN_CTRL_4	(VPL_SYSC_MMR_BASE + 0xAC)
#define SYSC_PAD_EN_CTRL_5	(VPL_SYSC_MMR_BASE + 0xB0)
#define SYSC_PAD_EN_CTRL_6	(VPL_SYSC_MMR_BASE + 0xB4)

#define SET_IO_LEVEL(pin, level) \
{ \
	if (level) { \
		readl(GPIOC_SET_DATA) |= pin; \
	} \
	else { \
		readl(GPIOC_CLEAR_DATA) |= pin; \
	} \
}\

#define SET_DATA_DIR(pin, dir) \
{ \
	u32 value; \
	value = readl(GPIOC_PIN_DIR); \
	if (dir == DATA_OUT) { \
		value |= pin; \
	} \
	else { \
		value &= ~pin; \
	} \
	writel(value, GPIOC_PIN_DIR); \
}\

#define GET_DATA_LEVEL(pin)		(readl(GPIOC_DATA_IN) & pin)

static void gpio_init(void)
{
	u32 value;
	u32 cnt_loop;
	printf("gpio init start!!\n");
	
#if 1 // SPI GPIO pinmux init
	// set pad enable
	value = readl(SYSC_PAD_EN_CTRL_3);
	value |= (SPI_CS | SPI_CLK | SPI_DATA | PANEL_RESET);
	writel(value, SYSC_PAD_EN_CTRL_3);

	// disable UARTC_1, GPIOC2_13, 14 and
	value = readl(SYSC_PAD_EN_CTRL_5);
	value &= ~(1 << 9 | 1 << 10 | 1 << 11 | 1 << 12 | 1 << 25 | 1 << 26 | 1 << 27 | 1 << 28 | 1 << 29 | 1 << 30 | 1 << 31);
	writel(value, SYSC_PAD_EN_CTRL_5);

	// disable UARTC_3, GPIOC2_9 and 10
	value = readl(SYSC_PAD_EN_CTRL_6);
	value &= ~(1 << 1);
	value |= (1 << 13 | 1 << 12 | 1 << 11);
	writel(value, SYSC_PAD_EN_CTRL_6);
#endif

	// set output direct
	value = readl(GPIOC_PIN_DIR);
	value |= (SPI_CS | SPI_CLK | SPI_DATA | PANEL_RESET);
	writel(value, GPIOC_PIN_DIR);

	// set level
	SET_IO_LEVEL(SPI_CS, 1);
	SET_IO_LEVEL(SPI_CLK, 1);
	SET_IO_LEVEL(SPI_DATA, 0);
	SET_IO_LEVEL(PANEL_RESET, 1);

	// set pull high
	value = readl(GPIOC_PULL_TYPE);
	value |= SPI_CS | SPI_CLK | SPI_DATA | PANEL_RESET;
	writel(value, GPIOC_PULL_TYPE);

	// set pull enable
	value = readl(GPIOC_PULL_EN);
	value |= SPI_CS | SPI_CLK | SPI_DATA | PANEL_RESET;
	writel(value, GPIOC_PULL_EN);

	udelay(30000);

	SET_IO_LEVEL(PANEL_RESET, 0);

	udelay(200000);
	
	SET_IO_LEVEL(PANEL_RESET, 1);

	udelay(30000);

	SET_IO_LEVEL(PANEL_RESET, 0);

	udelay(200000);

	SET_IO_LEVEL(PANEL_RESET, 1);
}
#elif (defined PANEL_ST7789) || (defined PANEL_ST7793)
// BackLight:   GPIOC0-10
#define BACK_LIGHT_EN       (1 << 10)

#define DATA_OUT            1
#define DATA_IN             0

#define GPIOC_DATA_OUT      (VPL_GPIOC_0_MMR_BASE + 0x04)
#define GPIOC_DATA_IN       (VPL_GPIOC_0_MMR_BASE + 0x08)
#define GPIOC_PIN_DIR       (VPL_GPIOC_0_MMR_BASE + 0x0C)
#define GPIOC_SET_DATA      (VPL_GPIOC_0_MMR_BASE + 0x14)
#define GPIOC_CLEAR_DATA    (VPL_GPIOC_0_MMR_BASE + 0x18)
#define GPIOC_PULL_EN       (VPL_GPIOC_0_MMR_BASE + 0x1C)
#define GPIOC_PULL_TYPE     (VPL_GPIOC_0_MMR_BASE + 0x20)

#define SYSC_PAD_EN_CTRL_4  (VPL_SYSC_MMR_BASE + 0xAC)
#define SYSC_PAD_EN_CTRL_5  (VPL_SYSC_MMR_BASE + 0xB0)
#define SYSC_PAD_EN_CTRL_6  (VPL_SYSC_MMR_BASE + 0xB4)

#define SET_IO_LEVEL(pin, level) \
{ \
    if (level) { \
        readl(GPIOC_SET_DATA) |= pin; \
    } \
    else { \
        readl(GPIOC_CLEAR_DATA) |= pin; \
    } \
}\

#define SET_DATA_DIR(pin, dir) \
{ \
    u32 value; \
    value = readl(GPIOC_PIN_DIR); \
    if (dir == DATA_OUT) { \
        value |= pin; \
    } \
    else { \
        value &= ~pin; \
    } \
    writel(value, GPIOC_PIN_DIR); \
}\

#define GET_DATA_LEVEL(pin)     (readl(GPIOC_DATA_IN) & pin)

static void gpio_init(void)
{
    u32 value;

    // set output direct
    value = readl(GPIOC_PIN_DIR); \
    value |= BACK_LIGHT_EN;
    writel(value, GPIOC_PIN_DIR);

    // set level
    SET_IO_LEVEL(BACK_LIGHT_EN, 1);

    // set pull high
    value = readl(GPIOC_PULL_TYPE);
    value |= BACK_LIGHT_EN;
    writel(value, GPIOC_PULL_TYPE);

    // set pull enable
    value = readl(GPIOC_PULL_EN);
    value |= BACK_LIGHT_EN;
    writel(value, GPIOC_PULL_EN);

    debug("0x%x = 0x%x\n", GPIOC_PIN_DIR, readl(GPIOC_PIN_DIR));
    debug("0x%x = 0x%x\n", GPIOC_DATA_OUT, readl(GPIOC_DATA_OUT));
    debug("0x%x = 0x%x\n", GPIOC_DATA_SET, readl(GPIOC_DATA_SET));
    debug("0x%x = 0x%x\n", GPIOC_DATA_CLEAR, readl(GPIOC_DATA_CLEAR));
    debug("0x%x = 0x%x\n", GPIOC_PULL_TYPE, readl(GPIOC_PULL_TYPE));
    debug("0x%x = 0x%x\n", GPIOC_PULL_EN, readl(GPIOC_PULL_EN));
}
#endif

#if (defined PANEL_NT39016)

#define W_CMD  (1 << 1)
#define R_DATA (0 << 1)

static void spi_write(unsigned char *data, int cbsize)
{
	unsigned char temp;
	unsigned int i, j;

	udelay(1);

	SET_IO_LEVEL(SPI_CLK, 0);

	udelay(1);

	for(i=0; i<cbsize; i++) {
		temp = data[i];
		for(j=0; j<8; j++) {
			if (temp & 0x80) {
				SET_IO_LEVEL(SPI_DATA, 1);
			}
			else {
				SET_IO_LEVEL(SPI_DATA, 0);
			}

			temp <<= 1;

			SET_IO_LEVEL(SPI_CLK, 1);

			udelay(1);

			SET_IO_LEVEL(SPI_CLK, 0);

			udelay(1);
		}
	}

	SET_IO_LEVEL(SPI_CLK, 1);

	udelay(1);
}

static int __nt39016_read(u8 cmd)
{
	unsigned int data = 0;
	int i;

	SET_IO_LEVEL(SPI_CS, 0);

	spi_write(&cmd, 1);

	SET_DATA_DIR(SPI_DATA, DATA_IN);

	for(i=0; i<8; i++) {
		data <<= 1;
		if (GET_DATA_LEVEL(SPI_DATA)) {
			data |= 1;
		}

		SET_IO_LEVEL(SPI_CLK, 1);

		udelay(1);

		SET_IO_LEVEL(SPI_CLK, 0);

		udelay(1);
	}

	SET_IO_LEVEL(SPI_CLK, 1);

	udelay(1);

	SET_IO_LEVEL(SPI_CS, 1);

	SET_DATA_DIR(SPI_DATA, DATA_OUT);

	printf("rd reg: 0x%x, data: 0x%x\n", cmd, data);
	return data;
}

static int __nt39016_write(u8 cmd, u8 data)
{
	u8 txbuf[2];

	txbuf[0] = (cmd << 2) | W_CMD;
	txbuf[1] = data & 0xFF;

	printf("wr reg: 0x%x, data: 0x%x\n", cmd, data);
	printf("spi data[0]:0x%x, spi data[1]:0x%x\n", txbuf[0], txbuf[1]);

	SET_IO_LEVEL(SPI_CS, 0);

	spi_write(txbuf, 2);

	SET_IO_LEVEL(SPI_CS, 1);

	return 0;
}

#elif (defined PANEL_ILI9341V) || (defined PANEL_ILI9342C) || (defined PANEL_ST7789V)

#define ILI_DCX_CMD  (0<<8)
#define ILI_DCX_DATA (1<<8)

static void __spi_9bit_write(u16 data)
{
    unsigned int i;

    SET_IO_LEVEL(SPI_CS, 0);
    SET_IO_LEVEL(SPI_CLK, 0);

    udelay(1);

    for(i=0; i<9; i++)
    {
        if (data & 0x100)
        {
            SET_IO_LEVEL(SPI_DATA, 1);
        }
        else
        {
            SET_IO_LEVEL(SPI_DATA, 0);
        }

        data <<= 1;

        SET_IO_LEVEL(SPI_CLK, 1);

        udelay(1);

        SET_IO_LEVEL(SPI_CLK, 0);

        udelay(1);
    }

    SET_IO_LEVEL(SPI_CLK, 0);
    SET_IO_LEVEL(SPI_CS, 1);

    udelay(1);
}

static int __spi_write_command(u8 cmd)
{
    u16 tmp;

    tmp = cmd | ILI_DCX_CMD;

    __spi_9bit_write(tmp);

    return 0;
}

static int __spi_write_data(u8 data)
{
    u16 tmp;

    tmp = data | ILI_DCX_DATA;

    __spi_9bit_write(tmp);

    return 0;
}

#elif (defined PANEL_RM68172_4INCH) || (defined PANEL_RM68172_5INCH)
static int spi_write_op(u16 waddr, u16 wr_data)
{
    unsigned int i;
    u16 first_tr = 0;
    u16 second_tr = 0;
    u16 third_tr = 0;
    u16 check_bit;
    u16 hight_c = 0x2000;
    u16 data_flag = 0x4000;

//    printf("spi_write_op 0x%x , addr 0x%x\n", wr_data,waddr);

    first_tr = (hight_c | (waddr >> 8));
    second_tr = waddr & 0x00FF;
    third_tr = data_flag | (wr_data & 0x00FF);

    SET_IO_LEVEL(SPI_CS, 0);
    SET_IO_LEVEL(SPI_CLK, 0);

    udelay(1);
    
    check_bit = 0x8000;
    
    for(i = 0; i < 16; i++)
    {
        if (first_tr & check_bit)
        {
            SET_IO_LEVEL(SPI_DATA, 1);
        }
        else
        {
            SET_IO_LEVEL(SPI_DATA, 0);
        }

        check_bit >>= 1;

        SET_IO_LEVEL(SPI_CLK, 1);

        udelay(1);

        SET_IO_LEVEL(SPI_CLK, 0);

        udelay(1);
    }

    check_bit = 0x8000;
    
    for(i = 0; i < 16; i++)
    {
        if (second_tr & check_bit)
        {
            SET_IO_LEVEL(SPI_DATA, 1);
        }
        else
        {
            SET_IO_LEVEL(SPI_DATA, 0);
        }

        check_bit >>= 1;

        SET_IO_LEVEL(SPI_CLK, 1);

        udelay(1);

        SET_IO_LEVEL(SPI_CLK, 0);

        udelay(1);
    }

    check_bit = 0x8000;
    
    for(i = 0; i < 16; i++)
    {
        if (third_tr & check_bit)
        {
            SET_IO_LEVEL(SPI_DATA, 1);
        }
        else
        {
            SET_IO_LEVEL(SPI_DATA, 0);
        }

        check_bit >>= 1;

        SET_IO_LEVEL(SPI_CLK, 1);

        udelay(1);

        SET_IO_LEVEL(SPI_CLK, 0);

        udelay(1);
    }

    SET_IO_LEVEL(SPI_CLK, 1);
    SET_IO_LEVEL(SPI_CS, 1);

    udelay(1);

    return 0;
}

static int spi_write_cmd(u16 waddr)
{
    unsigned int i;
    u16 first_tr = 0;
    u16 second_tr = 0;
    u16 check_bit;
    u16 hight_c = 0x2000;

    first_tr = hight_c | (waddr >> 8);
    second_tr = (waddr & 0x00FF);

//	printf("spi_write_cmd addr 0x%x , fir 0x%x, sec 0x%x\n",waddr, first_tr, second_tr);
	
    SET_IO_LEVEL(SPI_CS, 0);
    SET_IO_LEVEL(SPI_CLK, 0);

    udelay(1);
    
    check_bit = 0x8000;
    
    for(i = 0; i < 16; i++)
    {
        if (first_tr & check_bit)
        {
            SET_IO_LEVEL(SPI_DATA, 1);
        }
        else
        {
            SET_IO_LEVEL(SPI_DATA, 0);
        }

        check_bit >>= 1;

        SET_IO_LEVEL(SPI_CLK, 1);

        udelay(1);

        SET_IO_LEVEL(SPI_CLK, 0);

        udelay(1);
    }

    check_bit = 0x8000;
    
    for(i = 0; i < 16; i++)
    {
        if (second_tr & check_bit)
        {
            SET_IO_LEVEL(SPI_DATA, 1);
        }
        else
        {
            SET_IO_LEVEL(SPI_DATA, 0);
        }

        check_bit >>= 1;

        SET_IO_LEVEL(SPI_CLK, 1);

        udelay(1);

        SET_IO_LEVEL(SPI_CLK, 0);

        udelay(1);
    }

    SET_IO_LEVEL(SPI_CLK, 1);
    SET_IO_LEVEL(SPI_CS, 1);

    udelay(1);

    return 0;
}

#elif (defined PANEL_ST7789) || (defined PANEL_ST7793)

static int i80_cmd_wait_cmpt()
{
    unsigned long delay_times = 0;

    while(voc->dwI80Ctrl & 0x01)
    {
        if(delay_times >= 500)
            return -1;
        udelay(1000);
        delay_times++;
    }

    return 0;
}

static int i80_cmd_write(u32 index)
{
    volatile u32 reg_value;
    u32 write_len = 0;

    if (i80_cmd_wait_cmpt()) {
        printf("Wait CMD complete timeout\n");
        return -1;
    }

    voc->dwI80Timing = (voc->dwI80Timing & (~0xFFFF0000)) | (((index >> 8) & 0xFFFF) << 16);

//    printf("i80 timing %x\n", (u32)voc->dwI80Timing);

    voc->dwI80Ctrl = (voc->dwI80Ctrl & (~0x0000FF00)) | ((index & 0xFF) << 8);
    voc->dwI80Ctrl = (voc->dwI80Ctrl & (~0x000F0000)) | ((0     & 0x0F) << 16);
    voc->dwI80Ctrl = (voc->dwI80Ctrl & (~0x0000000E)) | ((0x03  & 0x07) << 1);
    voc->dwI80Ctrl = (voc->dwI80Ctrl & (~0x00400000)) | ((0     & 0x01) << 22);
    voc->dwI80Ctrl = (voc->dwI80Ctrl & (~0x00000001)) | ((0x01  & 0x01));

//    printf("i80 ctrl %x\n", (u32)voc->dwI80Ctrl);

    return i80_cmd_wait_cmpt();
}

static int i80_send_cmdargs(u32 cmd, int argc, u8 *argv)
{
    volatile u32 reg_value;
    int i = 0;

    if (i80_cmd_wait_cmpt()) {
        printf("Wait CMD complete timeout\n");
        return -1;
    }

    do {} while(!(voc->dwI80Timing & 0x0200));

    voc->dwI80WTCONData = argv[0];
//    printf("write (%x) %x\n", cmd, (u32)voc->dwI80WTCONData);

    voc->dwI80Timing = (voc->dwI80Timing & (~0xFFFF0000)) | (((cmd >> 8) & 0xFFFF) << 16);
//    printf("i80 timing %x\n", (u32)voc->dwI80Timing);

    voc->dwI80Ctrl = (voc->dwI80Ctrl & (~0x0000FF00)) | ((cmd   & 0xFF) << 8);
    voc->dwI80Ctrl = (voc->dwI80Ctrl & (~0x000F0000)) | ((0     & 0x0F) << 16);
    voc->dwI80Ctrl = (voc->dwI80Ctrl & (~0x0000000E)) | ((0x01  & 0x07) << 1);
    voc->dwI80Ctrl = (voc->dwI80Ctrl & (~0x00400000)) | ((0     & 0x01) << 22);
    voc->dwI80Ctrl = (voc->dwI80Ctrl & (~0x00000001)) | ((0x01  & 0x01));
//    printf("i80 ctrl %x\n", (u32)voc->dwI80Ctrl);

    for (i = 1; i < argc; i++) {
        if (i80_cmd_wait_cmpt()) {
            printf("Wait CMD complete timeout\n");
            return -1;
        }

        do {} while(!(voc->dwI80Timing & 0x0200));
        voc->dwI80WTCONData = argv[i];
//        printf("write (%x) %x\n", cmd, (u32)voc->dwI80WTCONData);

        voc->dwI80Timing = (voc->dwI80Timing & (~0xFFFF0000)) | (((cmd >> 8) & 0xFFFF) << 16);
//        printf("i80 timing %x\n", (u32)voc->dwI80Timing);

        voc->dwI80Ctrl = (voc->dwI80Ctrl & (~0x0000FF00)) | ((cmd   & 0xFF) << 8);
        voc->dwI80Ctrl = (voc->dwI80Ctrl & (~0x000F0000)) | ((0     & 0x0F) << 16);
        voc->dwI80Ctrl = (voc->dwI80Ctrl & (~0x0000000E)) | ((0x01  & 0x07) << 1);
        voc->dwI80Ctrl = (voc->dwI80Ctrl & (~0x00400000)) | ((0x01  & 0x01) << 22);
        voc->dwI80Ctrl = (voc->dwI80Ctrl & (~0x00000001)) | ((0x01  & 0x01));
//        printf("i80 ctrl %x\n", (u32)voc->dwI80Ctrl);
    }
    return i80_cmd_wait_cmpt();
}
#endif

static void vpl_voc_pllc_config(u32 plldiv, u32 voc_clk_div, bool fast_slewrate)
{
	int limit = 200;
	u32 sys_voc;

	// VDAC clock enable control.
	writel(readl(VPL_SYSC_MMR_BASE + SYSC_MMR_CLKEN_CTRL1) | (0x01 << 14), VPL_SYSC_MMR_BASE + SYSC_MMR_CLKEN_CTRL1);

	writel(plldiv, VPL_PLLC_MMR_BASE + PLL4_MMR_DIV);
	writel(0x01, VPL_PLLC_MMR_BASE + PLL4_MMR_CTRL);

	/*Wait for PLL Update Done and Lock*/
	while(limit--) {
		if ((readl(VPL_PLLC_MMR_BASE + PLL4_MMR_CTRL) & 0x31) == 0x30)
			break;
		udelay(1000);
	}
	if (limit < 0)
		printf("VPL_VOC: Can't lock stable PLL clock output for a limited time\n");

	/* Set SYSC VOC Ctrl */
	sys_voc = readl(VPL_SYSC_MMR_BASE + SYSC_MMR_VOC_CTRL);
	sys_voc &= ~0x110310F;

	/* Set Pixel Clock Divider */
	sys_voc |= voc_clk_div;

	/* set srgb/I80 output and div3 clock source */
#if (defined PANEL_NT39016) || (defined PANEL_ILI9341V) || (defined PANEL_ILI9342C) || (defined PANEL_ST7789V)
	sys_voc |= 0x100;
#elif (defined PANEL_ST7789) || (defined PANEL_ST7793)
	sys_voc |= 0x100100;
#endif

	/* Set pad slew rate for high frequency clock */
	if (fast_slewrate)
		sys_voc |= 0x1000000;

	sys_voc |= (modes.pclk_pol << 12);

	/* Restart VOC to sync clock phase */
	readl(VPL_SYSC_MMR_BASE + SYSC_MMR_VOC_CTRL) = sys_voc | 0x10;
	readl(VPL_SYSC_MMR_BASE + SYSC_MMR_VOC_CTRL) = sys_voc;

	printf("pll4_div(0x%x) = 0x%x\n", VPL_PLLC_MMR_BASE + PLL4_MMR_DIV, readl(VPL_PLLC_MMR_BASE + PLL4_MMR_DIV));
	printf("sysc_voc(0x%x) = 0x%x\n", VPL_SYSC_MMR_BASE + SYSC_MMR_VOC_CTRL, readl(VPL_SYSC_MMR_BASE + SYSC_MMR_VOC_CTRL));

#if 1   //vdac move to standalone
	writel(0x01, VPL_VDAC_MMR_BASE + VDAC_BG_CTRL);
	writel(0x01, VPL_VDAC_MMR_BASE + VDAC_LDO_CTRL);

	limit = 200;
	while(limit--) {
		if (readl(VPL_VDAC_MMR_BASE + VDAC_BG_CTRL) & 0x80000000)
			break;
		udelay(1000);
	}

	if (limit < 0)
		debug("Dac BG not good\n");
#endif
}

static void vpl_voc_set_pclk(u32 dwPixelClk)
{
	u32 pclk_out;
	u32 pll_fb, pll_ref, pll_div, voc_div;
	int ret;

	u32 dwPClkDiv = 0;
	u32 dwPLLRatio;

	ret = voc_compute_pll_and_div(dwPixelClk, &pclk_out, &pll_fb, &pll_ref, &pll_div, &voc_div);

	if (ret) {
		debug("Can't find PLL parameters for required Pixel Clock %d\n", dwPixelClk);
		return;
	}

	dwPLLRatio = ((pll_fb - 1) << 16) | ((pll_div - 1) << 8) | (pll_ref - 1);
	dwPClkDiv = (voc_div - 1) & 0xF;

	debug("request PCLK: %d\n", dwPixelClk);
	debug("actual  PCLK: %d\n", pclk_out);
	debug("pll_fb %d\n", pll_fb);
	debug("pll_ref %d\n", pll_ref);
	debug("pll_div %d\n", pll_div);
	debug("voc_div %d\n", voc_div);
	debug("dwPLLRatio %08x\n", dwPLLRatio);
	debug("dwPClkDiv %08x\n", dwPClkDiv);

	/*TODO: set slew rate by frequency */
	vpl_voc_pllc_config(dwPLLRatio, dwPClkDiv, 0);
}

static int update_display_mode(void)
{
	u32 dwValue;
	u32 dwOutWidth, dwOutHeight;

	dwOutWidth = modes.hactive + modes.hback_porch + modes.hfront_porch + modes.hsync_len;
	dwOutHeight = modes.vactive + modes.vback_porch + modes.vfront_porch + modes.vsync_len;

	/* Total size */
	voc->dwOutSize = (dwOutHeight<<16) | dwOutWidth;

	/* Active size */
	voc->dwInSize = (modes.vactive<<16) | modes.hactive;

	/* HSync */
	voc->dwHSyncCtrl = (modes.hsync_act_pol<<31) | (modes.hfront_porch << 12) | modes.hback_porch | 0x800;

	/* VSync */
	//set_field(dwValue, VOC_VSYNC_CTRL_INTERLACED, timing->bInterlaced);
	voc->dwVSyncCtrl = (modes.vsync_act_pol<<31) | (modes.vfront_porch << 12) | modes.vback_porch | 0x800;

	/* Blank Pol*/
	voc->dwCtrl |= modes.de_act_pol;

	/* PClk Pol*/
#if 0
    dwValue = readl(VPL_SYSC_MMR_BASE + SYSC_MMR_VOC_CTRL);
	dwValue &= ~0x300F;
	dwValue |= (modes.pclk_pol << 12);

	// VOC clock source selection.
#if (defined PANEL_NT39016) || (defined PANEL_ILI9341V) || (defined PANEL_ILI9342C) || (defined PANEL_ST7789V)
	    dwValue |= 0x100;
#elif (defined PANEL_ST7789) || (defined PANEL_ST7793)
	    dwValue |= 0x100100;
#endif

	// VOC clock frequency divider. (PLL 4 Output Clock Frequency) / 4(VOC_CLK_DIV+1)
	dwValue |= 0x09;

	writel(dwValue, VPL_SYSC_MMR_BASE + SYSC_MMR_VOC_CTRL);
	debug("sysc_voc(0x%x) = 0x%x\n", VPL_SYSC_MMR_BASE + SYSC_MMR_VOC_CTRL, readl(VPL_SYSC_MMR_BASE + SYSC_MMR_VOC_CTRL));
#else
	vpl_voc_set_pclk(modes.pixelclock);
#endif

	return 0;
}

static int lcd_init(void)
{
	u32 val;
	u32 panel_rotate = 0;

#ifdef CONFIG_PANEL_ROTATE
    char *s;
    if ((s = getenv ("panel_rotate")) != NULL)
    {
        panel_rotate = simple_strtoul (s, NULL, 10);
    }
#endif

	/* Setup LCD Panel */
#if (defined PANEL_NT39016) || (defined PANEL_ILI9341V) || (defined PANEL_ILI9342C) || (defined PANEL_ST7789) || (defined PANEL_ST7793) || (defined PANEL_ST7789V) || (defined PANEL_RM68172_4INCH) || (defined PANEL_RM68172_5INCH)
	gpio_init();
#endif

#if (defined PANEL_NT39016)
	/* Reset */
	__nt39016_write(0x0, 0x06);
	__nt39016_read(0x0);

	/* Test Pattern Auto Run Mode */
	//__nt39016_write(0x0, 0xf7);
#if 1
	/* HV Mode */
	__nt39016_write(0x3, 0xc8);
	__nt39016_write(0x4, 0x48);
#else
	/* DE Mode */
	__nt39016_write(&spi->dev, 0x3, 0xc9);
#endif

#elif (defined PANEL_ILI9341V)

    __spi_write_command(0x01);
    udelay(120000);

    //Power control B
    __spi_write_command(0xCF);
    __spi_write_data(0x00);
    __spi_write_data(0xC1);
    __spi_write_data(0x30);

    __spi_write_command(0xED);
    __spi_write_data(0x64);
    __spi_write_data(0x03);
    __spi_write_data(0x12);
    __spi_write_data(0x81);

    __spi_write_command(0xE8);
    __spi_write_data(0x85);
    __spi_write_data(0x00);
    __spi_write_data(0x78);

    __spi_write_command(0xCB);
    __spi_write_data(0x39);
    __spi_write_data(0x2C);
    __spi_write_data(0x00);
    __spi_write_data(0x34);
    __spi_write_data(0x02);

    __spi_write_command(0xC0);     //Power control
    __spi_write_data(0x1B);        //VRH[5:0]

    __spi_write_command(0xC1);     //Power control
    __spi_write_data(0x12);        //SAP[2:0];BT[3:0]

    __spi_write_command(0xC5);     //VCM control
    __spi_write_data(0x32);
    __spi_write_data(0x3C);

    __spi_write_command(0xC7);
    __spi_write_data(0x9E);//0x9A

    __spi_write_command(0xE0);     //Set Gamma
    __spi_write_data(0x0F);
    __spi_write_data(0x1D);
    __spi_write_data(0x1A);
    __spi_write_data(0x0A);
    __spi_write_data(0x0D);
    __spi_write_data(0x07);
    __spi_write_data(0x49);
    __spi_write_data(0x66);
    __spi_write_data(0x3B);
    __spi_write_data(0x07);
    __spi_write_data(0x11);
    __spi_write_data(0x01);
    __spi_write_data(0x09);
    __spi_write_data(0x05);
    __spi_write_data(0x04);

    __spi_write_command(0XE1);     //Set Gamma
    __spi_write_data(0x00);
    __spi_write_data(0x18);
    __spi_write_data(0x1D);
    __spi_write_data(0x02);
    __spi_write_data(0x0F);
    __spi_write_data(0x04);
    __spi_write_data(0x36);
    __spi_write_data(0x13);
    __spi_write_data(0x4C);
    __spi_write_data(0x07);
    __spi_write_data(0x13);
    __spi_write_data(0x0F);
    __spi_write_data(0x2E);
    __spi_write_data(0x2F);
    __spi_write_data(0x05);

    __spi_write_command(0x3A);
    __spi_write_data(0x66);

    __spi_write_command(0x36);     // Memory Access Control
    __spi_write_data(0x00);

    __spi_write_command(0xB1);     // Frame Rate Control
    __spi_write_data(0x00);
    __spi_write_data(0x15);

    __spi_write_command(0xB6);     // Display Function Control
    __spi_write_data(0x0A);
    __spi_write_data(0xA2);

    __spi_write_command(0xF2);     // 3Gamma Function Disable
    __spi_write_data(0x00);

    __spi_write_command(0x26);     //Gamma curve selected
    __spi_write_data(0x01);


    __spi_write_command(0xB0);     // Frame Rate Control
#if 0
    /* DE Mode */
    __spi_write_data(0xC0);
#else
    /* SYNC Mode */
    __spi_write_data(0xE0);

    __spi_write_command(0xB5);
    __spi_write_data(0x04);
    __spi_write_data(0x04);        /* VBP = Vsync + Back Porch, not only back porch, check timing diagram in 7.5.7 */
    __spi_write_data(0x0C);        /* 12 DOTCLK */
    __spi_write_data(0x21);        /* HBP = Hsync + Back Porch,  12 + 21 DOTCLK */
#endif

    __spi_write_command(0XF6);     //Set
    __spi_write_data(0x01);
    __spi_write_data(0x00);
    __spi_write_data(0x07);

    __spi_write_command(0x11);     //Exit Sleep
    udelay(120000);
    __spi_write_command(0x29);     //display on
    __spi_write_command(0x2c);

#elif (defined PANEL_ILI9342C)
	/* SW reset */
	__spi_write_command(0x01);
	udelay(120000); /* 5msec ~ 120msec*/

	/* Software Reset */
	__spi_write_command(0xC8);
	__spi_write_data(0xFF);
	__spi_write_data(0x93);
	__spi_write_data(0x42);

	/* Memory Access Control */
#if 0
	__spi_write_command(0x36);
	__spi_write_data(0x08);
#else
	if (panel_rotate)
	{
        __spi_write_command(0x36);   // Memory Access Control
        __spi_write_data(0xC8);
	}
	else
	{
        __spi_write_command(0x36);    // Memory Access Control
        __spi_write_data(0x08);
	}
#endif

	/* RGB Interface Signal Control */
#if 1
	/* SYNC Mode */
	__spi_write_command(0xB0);  //same as 9342?
	__spi_write_data(0xE2);
#else
	/* DE Mode */
	__spi_write_command(0xB0);  //same as 9342?
	__spi_write_data(0xC2);
#endif
	/* Blanking Porch Control */
	__spi_write_command(0xB5);
	__spi_write_data(0x04);
	__spi_write_data(0x04); /* VBP = Vsync + Back Porch, not only back porch, check timing diagram in 7.5.7 */
	__spi_write_data(0x0C); /* 12 DOTCLK */
	__spi_write_data(0x21); /* HBP = Hsync + Back Porch,  12 + 21 DOTCLK */

	/* Interface Control */
	__spi_write_command(0xF6);
	__spi_write_data(0x01);
	__spi_write_data(0x00);
	__spi_write_data(0x07);

	/* Display Inversion Control */
	__spi_write_command(0xB4);
	__spi_write_data(0x02);

	/* COLMOD: Pixel Format Set */
	__spi_write_command(0x3A);
	__spi_write_data(0x55);

	/* Power Control 1 */
	__spi_write_command(0xC0);
	__spi_write_data(0x0F);
	__spi_write_data(0x0F);

	/* VCOM Control 1 */
	__spi_write_command(0xC5);
	__spi_write_data(0xDB);

	/* Positive Gamma Correction */
	__spi_write_command(0xE0); //Set Gamma
	__spi_write_data(0x00);
	__spi_write_data(0x05);
	__spi_write_data(0x08);
	__spi_write_data(0x02);
	__spi_write_data(0x1A);
	__spi_write_data(0x0C);
	__spi_write_data(0x42);
	__spi_write_data(0x7A);
	__spi_write_data(0x54);
	__spi_write_data(0x08);
	__spi_write_data(0x0D);
	__spi_write_data(0x0C);
	__spi_write_data(0x23);
	__spi_write_data(0x25);
	__spi_write_data(0x0F);

	/* Negative Gamma Correction */
	__spi_write_command(0xE1); //Set Gamma
	__spi_write_data(0x00);
	__spi_write_data(0x29);
	__spi_write_data(0x2F);
	__spi_write_data(0x03);
	__spi_write_data(0x0F);
	__spi_write_data(0x05);
	__spi_write_data(0x42);
	__spi_write_data(0x55);
	__spi_write_data(0x53);
	__spi_write_data(0x06);
	__spi_write_data(0x0F);
	__spi_write_data(0x0C);
	__spi_write_data(0x38);
	__spi_write_data(0x3A);
	__spi_write_data(0x0F);

	//__spi_write_command(0x21);//Inversion

	__spi_write_command(0x11); //Exit SleepDelay(120);
	udelay(120000);

	__spi_write_command(0x29); //display on

#elif (defined PANEL_ST7789)

    /* Reset I80 internal satus machine and count, DOESN'T clear MMR */
    voc->dwI80Ctrl |= (1 << 7);
    udelay(1);
//    printf("i80 reset\n");
    do {} while (voc->dwI80Ctrl & 0x80);
//    printf("i80 reset done\n");
    /* Set Pixel Sequence */
    voc->dwCtrl &= (~0x03000000);

    /* Set TE mode */
//    voc->dwI80Ctrl |= (1 << 4);

    /* Set Read timing */
    voc->dwI80Timing |= (3 << 4);
    voc->dwI80Timing |= 3;
    voc->dwI80Timing |= (0 << 12);

    /* Set CMD propertise */
    voc->dwI80Ctrl |= (0 << 20);

    i80_cmd_write(0x01);
    udelay(120000);
    //--------------------------------------------------//
    i80_cmd_write(0x11);
    udelay(120000);
    //-------------Display and color format setting---------------//

    for (val = 0; val < ARRAY_SIZE(st7789s_init_tbl); val++)
    {
        i80_send_cmdargs(st7789s_init_tbl[val].cmd, st7789s_init_tbl[val].num, st7789s_init_tbl[val].value);
    }

    i80_cmd_write(0x29);

    voc->dwI80Ctrl = (voc->dwI80Ctrl & (~0x0000FF00)) | ((0x2c  & 0xFF) << 8);
    voc->dwI80Ctrl = (voc->dwI80Ctrl & (~0x000F0000)) | ((0     & 0x0F) << 16);
    voc->dwI80Ctrl = (voc->dwI80Ctrl & (~0x0000000E)) | ((0x05  & 0x07) << 1);
    voc->dwI80Ctrl = (voc->dwI80Ctrl & (~0x00400000)) | ((0     & 0x01) << 22);
    voc->dwI80Ctrl = (voc->dwI80Ctrl & (~0x00000040)) | ((0x01  & 0x01) << 6);
//    printf("start %x\n", voc->dwI80Ctrl);
    voc->dwI80Ctrl = (voc->dwI80Ctrl & (~0x00000001)) | ((0x01  & 0x01));

#elif (defined PANEL_ST7793)

    /* Reset I80 internal satus machine and count, DOESN'T clear MMR */
    voc->dwI80Ctrl |= (1 << 7);
    udelay(1);
//    printf("i80 reset\n");
    do {} while (voc->dwI80Ctrl & 0x80);
//    printf("i80 reset done\n");
    /* Set Pixel Sequence */
    voc->dwCtrl &= (~0x03000000);

    /* Set TE mode */
    voc->dwI80Ctrl |= (1 << 4);

    /* Set Read timing */
    voc->dwI80Timing |= (3 << 4);
    voc->dwI80Timing |= 3;
    voc->dwI80Timing |= (0 << 12);

    /* Set CMD propertise */
    voc->dwI80Ctrl |= (1 << 20);


    for (val = 0; val < ARRAY_SIZE(st7793_init_tbl); val++)
    {
        i80_send_cmdargs(st7793_init_tbl[val].cmd, st7793_init_tbl[val].num, st7793_init_tbl[val].value);
    }

    voc->dwI80Timing = (voc->dwI80Timing & (~0xFFFF0000)) | (((0x202 >> 8) & 0xFFFF) << 16);

    voc->dwI80Ctrl = (voc->dwI80Ctrl & (~0x0000FF00)) | ((0x202 & 0xFF) << 8);
    voc->dwI80Ctrl = (voc->dwI80Ctrl & (~0x000F0000)) | ((0     & 0x0F) << 16);
    voc->dwI80Ctrl = (voc->dwI80Ctrl & (~0x0000000E)) | ((0x05  & 0x07) << 1);
    voc->dwI80Ctrl = (voc->dwI80Ctrl & (~0x00400000)) | ((0     & 0x01) << 22);
    voc->dwI80Ctrl = (voc->dwI80Ctrl & (~0x00000040)) | ((0x01  & 0x01) << 6);
//    printf("start %x\n", voc->dwI80Ctrl);
    voc->dwI80Ctrl = (voc->dwI80Ctrl & (~0x00000001)) | ((0x01  & 0x01));
#elif (defined PANEL_ST7789V)
	// SW reset
	//__spi_write_command(0x01);
	//udelay(120000); //Delay 120ms

	// Sleep Out
	__spi_write_command(0x11);
	udelay(120000); //Delay 120ms

	// back porch setting
	__spi_write_command(0xb2);
	__spi_write_data(0x08);
	__spi_write_data(0x08);
	__spi_write_data(0x00);
	__spi_write_data(0x33);
	__spi_write_data(0x33);

	// setting VGH/VGL
	__spi_write_command(0xb7);
	__spi_write_data(0x35);

	// Power setting
	__spi_write_command(0xbb); //vcom setting
	__spi_write_data(0x20);

	__spi_write_command(0xc3); //VRH set
	__spi_write_data(0x0b);

	__spi_write_command(0xc4); //VDV SET
	__spi_write_data(0x20);

	__spi_write_command(0xc6); //INVERSION SELECTION
	__spi_write_data(0x0f);

	__spi_write_command(0xd0); //POWER CONTROL
	__spi_write_data(0xa4);
	__spi_write_data(0xa2);

	// gamma setting
	__spi_write_command(0xe0);
	__spi_write_data(0xd0);
	__spi_write_data(0x00);
	__spi_write_data(0x02);
	__spi_write_data(0x07);
	__spi_write_data(0x07);
	__spi_write_data(0x19);
	__spi_write_data(0x2e);
	__spi_write_data(0x54);
	__spi_write_data(0x41);
	__spi_write_data(0x2d);
	__spi_write_data(0x17);
	__spi_write_data(0x18);
	__spi_write_data(0x14);
	__spi_write_data(0x18);

	__spi_write_command(0xe1);
	__spi_write_data(0xd0);
	__spi_write_data(0x00);
	__spi_write_data(0x02);
	__spi_write_data(0x07);
	__spi_write_data(0x04);
	__spi_write_data(0x24);
	__spi_write_data(0x2c);
	__spi_write_data(0x44);
	__spi_write_data(0x42);
	__spi_write_data(0x1c);
	__spi_write_data(0x1a);
	__spi_write_data(0x17);
	__spi_write_data(0x15);
	__spi_write_data(0x18);

	// equalize time control
	__spi_write_command(0xe9);
	__spi_write_data(0x08);
	__spi_write_data(0x08);
	__spi_write_data(0x04);

	__spi_write_command(0xb8);
	__spi_write_data(0x25);
	__spi_write_data(0x15);
	__spi_write_data(0x20);

	// gate control
	__spi_write_command(0xe4);
	__spi_write_data(0x27);
	__spi_write_data(0x00);
	__spi_write_data(0x10); //Gate scan direction is 0 to 319
	//__spi_write_data(0x11); //Gate scan direction is 319 to 0

	// Command 2 Enable
	__spi_write_command(0xdf);
	__spi_write_data(0x5a);
	__spi_write_data(0x69);
	__spi_write_data(0x02);
	__spi_write_data(0x01);

	// Mirror
	if (panel_rotate) {
		// adjust back porch
		voc->dwOutSize += (0x58 << 16);
		__spi_write_command(0x36);
		__spi_write_data(0x14); //0xC0
	}
	else {
		__spi_write_command(0x36);
		__spi_write_data(0x00);
	}

	// Interface Pixel Format
	__spi_write_command(0x3a);  // 262k
	__spi_write_data(0x66);

	// RAM Control
	__spi_write_command(0xb0);
	__spi_write_data(0x11);
	__spi_write_data(0xC4);

	// RGB Interface Control
	__spi_write_command(0xb1);
	__spi_write_data(0x60 | 0x80); // without ram.
	__spi_write_data(0x10); // RGB interface Vsync back porch setting
	__spi_write_data(0x10); // RGB interface Hsync back porch setting

	__spi_write_command(0x21);
	__spi_write_command(0x29); //Display on

#elif (defined PANEL_RM68172_4INCH)
	spi_write_op(0xF000,0x55);
	spi_write_op(0xF001,0xAA);
	spi_write_op(0xF002,0x52);
	spi_write_op(0xF003,0x08);
	spi_write_op(0xF004,0x02);

	spi_write_op(0xF600,0x60);
	spi_write_op(0xF601,0x40);

	spi_write_op(0xFE00,0x01);
	spi_write_op(0xFE01,0x80);
	spi_write_op(0xFE02,0x09);
	spi_write_op(0xFE03,0x09);

	udelay(10);

	spi_write_op(0xF000,0x55);
	spi_write_op(0xF001,0xAA);
	spi_write_op(0xF002,0x52);
	spi_write_op(0xF003,0x08);
	spi_write_op(0xF004,0x01);

	spi_write_op(0xB000,0x0D);
	spi_write_op(0xB001,0x0D);
	spi_write_op(0xB002,0x0D);

	spi_write_op(0xB100,0x0D);
	spi_write_op(0xB101,0x0D);
	spi_write_op(0xB102,0x0D);

	spi_write_op(0xB200,0x00);

	spi_write_op(0xB500,0x08);
	spi_write_op(0xB501,0x08);
	spi_write_op(0xB502,0x08);

	spi_write_op(0xB600,0x34);
	spi_write_op(0xB601,0x34);
	spi_write_op(0xB602,0x34);

	spi_write_op(0xB700,0x44);
	spi_write_op(0xB701,0x44);
	spi_write_op(0xB702,0x44);

	spi_write_op(0xB800,0x24);
	spi_write_op(0xB801,0x24);
	spi_write_op(0xB802,0x24);

	spi_write_op(0xB900,0x34);
	spi_write_op(0xB901,0x34);
	spi_write_op(0xB902,0x34);

	spi_write_op(0xBA00,0x14);
	spi_write_op(0xBA01,0x14);
	spi_write_op(0xBA02,0x14);

	spi_write_op(0xBC00,0x00);
	spi_write_op(0xBC01,0xA8);
	spi_write_op(0xBC02,0x00);

	spi_write_op(0xBD00,0x00);
	spi_write_op(0xBD01,0xA8);
	spi_write_op(0xBD02,0x00);

	spi_write_op(0xBE00,0x00);
	spi_write_op(0xBE01,0x48);

	spi_write_op(0xD100,0x00);
	spi_write_op(0xD101,0x00);
	spi_write_op(0xD102,0x00);
	spi_write_op(0xD103,0x11);
	spi_write_op(0xD104,0x00);
	spi_write_op(0xD105,0x2E);
	spi_write_op(0xD106,0x00);
	spi_write_op(0xD107,0x47);
	spi_write_op(0xD108,0x00);
	spi_write_op(0xD109,0x5C);
	spi_write_op(0xD10A,0x00);
	spi_write_op(0xD10B,0x7E);
	spi_write_op(0xD10C,0x00);
	spi_write_op(0xD10D,0x9C);
	spi_write_op(0xD10E,0x00);
	spi_write_op(0xD10F,0xCD);
	spi_write_op(0xD110,0x00);
	spi_write_op(0xD111,0xF6);
	spi_write_op(0xD112,0x01);
	spi_write_op(0xD113,0x36);
	spi_write_op(0xD114,0x01);
	spi_write_op(0xD115,0x6A);
	spi_write_op(0xD116,0x01);
	spi_write_op(0xD117,0xBD);
	spi_write_op(0xD118,0x01);
	spi_write_op(0xD119,0xFF);
	spi_write_op(0xD11A,0x02);
	spi_write_op(0xD11B,0x01);
	spi_write_op(0xD11C,0x02);
	spi_write_op(0xD11D,0x3B);
	spi_write_op(0xD11E,0x02);
	spi_write_op(0xD11F,0x75);
	spi_write_op(0xD120,0x02);
	spi_write_op(0xD121,0x99);
	spi_write_op(0xD122,0x02);
	spi_write_op(0xD123,0xC8);
	spi_write_op(0xD124,0x02);
	spi_write_op(0xD125,0xE8);
	spi_write_op(0xD126,0x03);
	spi_write_op(0xD127,0x11);
	spi_write_op(0xD128,0x03);
	spi_write_op(0xD129,0x2B);
	spi_write_op(0xD12A,0x03);
	spi_write_op(0xD12B,0x48);
	spi_write_op(0xD12C,0x03);
	spi_write_op(0xD12D,0x59);
	spi_write_op(0xD12E,0x03);
	spi_write_op(0xD12F,0x65);
	spi_write_op(0xD130,0x03);
	spi_write_op(0xD131,0x71);
	spi_write_op(0xD132,0x03);
	spi_write_op(0xD133,0xFF);

	spi_write_op(0xD200,0x00);
	spi_write_op(0xD201,0x00);
	spi_write_op(0xD202,0x00);
	spi_write_op(0xD203,0x11);
	spi_write_op(0xD204,0x00);
	spi_write_op(0xD205,0x2E);
	spi_write_op(0xD206,0x00);
	spi_write_op(0xD207,0x47);
	spi_write_op(0xD208,0x00);
	spi_write_op(0xD209,0x5C);
	spi_write_op(0xD20A,0x00);
	spi_write_op(0xD20B,0x7E);
	spi_write_op(0xD20C,0x00);
	spi_write_op(0xD20D,0x9C);
	spi_write_op(0xD20E,0x00);
	spi_write_op(0xD20F,0xCD);
	spi_write_op(0xD210,0x00);
	spi_write_op(0xD211,0xF6);
	spi_write_op(0xD212,0x01);
	spi_write_op(0xD213,0x36);
	spi_write_op(0xD214,0x01);
	spi_write_op(0xD215,0x6A);
	spi_write_op(0xD216,0x01);
	spi_write_op(0xD217,0xBD);
	spi_write_op(0xD218,0x01);
	spi_write_op(0xD219,0xFF);
	spi_write_op(0xD21A,0x02);
	spi_write_op(0xD21B,0x01);
	spi_write_op(0xD21C,0x02);
	spi_write_op(0xD21D,0x3B);
	spi_write_op(0xD21E,0x02);
	spi_write_op(0xD21F,0x75);
	spi_write_op(0xD220,0x02);
	spi_write_op(0xD221,0x99);
	spi_write_op(0xD222,0x02);
	spi_write_op(0xD223,0xC8);
	spi_write_op(0xD224,0x02);
	spi_write_op(0xD225,0xE8);
	spi_write_op(0xD226,0x03);
	spi_write_op(0xD227,0x11);
	spi_write_op(0xD228,0x03);
	spi_write_op(0xD229,0x2B);
	spi_write_op(0xD22A,0x03);
	spi_write_op(0xD22B,0x48);
	spi_write_op(0xD22C,0x03);
	spi_write_op(0xD22D,0x59);
	spi_write_op(0xD22E,0x03);
	spi_write_op(0xD22F,0x65);
	spi_write_op(0xD230,0x03);
	spi_write_op(0xD231,0x71);
	spi_write_op(0xD232,0x03);
	spi_write_op(0xD233,0xFF);

	spi_write_op(0xD300,0x00);
	spi_write_op(0xD301,0x00);
	spi_write_op(0xD302,0x00);
	spi_write_op(0xD303,0x11);
	spi_write_op(0xD304,0x00);
	spi_write_op(0xD305,0x2E);
	spi_write_op(0xD306,0x00);
	spi_write_op(0xD307,0x47);
	spi_write_op(0xD308,0x00);
	spi_write_op(0xD309,0x5C);
	spi_write_op(0xD30A,0x00);
	spi_write_op(0xD30B,0x7E);
	spi_write_op(0xD30C,0x00);
	spi_write_op(0xD30D,0x9C);
	spi_write_op(0xD30E,0x00);
	spi_write_op(0xD30F,0xCD);
	spi_write_op(0xD310,0x00);
	spi_write_op(0xD311,0xF6);
	spi_write_op(0xD312,0x01);
	spi_write_op(0xD313,0x36);
	spi_write_op(0xD314,0x01);
	spi_write_op(0xD315,0x6A);
	spi_write_op(0xD316,0x01);
	spi_write_op(0xD317,0xBD);
	spi_write_op(0xD318,0x01);
	spi_write_op(0xD319,0xFF);
	spi_write_op(0xD31A,0x02);
	spi_write_op(0xD31B,0x01);
	spi_write_op(0xD31C,0x02);
	spi_write_op(0xD31D,0x3B);
	spi_write_op(0xD31E,0x02);
	spi_write_op(0xD31F,0x75);
	spi_write_op(0xD320,0x02);
	spi_write_op(0xD321,0x99);
	spi_write_op(0xD322,0x02);
	spi_write_op(0xD323,0xC8);
	spi_write_op(0xD324,0x02);
	spi_write_op(0xD325,0xE8);
	spi_write_op(0xD326,0x03);
	spi_write_op(0xD327,0x11);
	spi_write_op(0xD328,0x03);
	spi_write_op(0xD329,0x2B);
	spi_write_op(0xD32A,0x03);
	spi_write_op(0xD32B,0x48);
	spi_write_op(0xD32C,0x03);
	spi_write_op(0xD32D,0x59);
	spi_write_op(0xD32E,0x03);
	spi_write_op(0xD32F,0x65);
	spi_write_op(0xD330,0x03);
	spi_write_op(0xD331,0x71);
	spi_write_op(0xD332,0x03);
	spi_write_op(0xD333,0xFF);

	spi_write_op(0xD400,0x00);
	spi_write_op(0xD401,0x00);
	spi_write_op(0xD402,0x00);
	spi_write_op(0xD403,0x11);
	spi_write_op(0xD404,0x00);
	spi_write_op(0xD405,0x2E);
	spi_write_op(0xD406,0x00);
	spi_write_op(0xD407,0x47);
	spi_write_op(0xD408,0x00);
	spi_write_op(0xD409,0x5C);
	spi_write_op(0xD40A,0x00);
	spi_write_op(0xD40B,0x7E);
	spi_write_op(0xD40C,0x00);
	spi_write_op(0xD40D,0x9C);
	spi_write_op(0xD40E,0x00);
	spi_write_op(0xD40F,0xCD);
	spi_write_op(0xD410,0x00);
	spi_write_op(0xD411,0xF6);
	spi_write_op(0xD412,0x01);
	spi_write_op(0xD413,0x36);
	spi_write_op(0xD414,0x01);
	spi_write_op(0xD415,0x6A);
	spi_write_op(0xD416,0x01);
	spi_write_op(0xD417,0xBD);
	spi_write_op(0xD418,0x01);
	spi_write_op(0xD419,0xFF);
	spi_write_op(0xD41A,0x02);
	spi_write_op(0xD41B,0x01);
	spi_write_op(0xD41C,0x02);
	spi_write_op(0xD41D,0x3B);
	spi_write_op(0xD41E,0x02);
	spi_write_op(0xD41F,0x75);
	spi_write_op(0xD420,0x02);
	spi_write_op(0xD421,0x99);
	spi_write_op(0xD422,0x02);
	spi_write_op(0xD423,0xC8);
	spi_write_op(0xD424,0x02);
	spi_write_op(0xD425,0xE8);
	spi_write_op(0xD426,0x03);
	spi_write_op(0xD427,0x11);
	spi_write_op(0xD428,0x03);
	spi_write_op(0xD429,0x2B);
	spi_write_op(0xD42A,0x03);
	spi_write_op(0xD42B,0x48);
	spi_write_op(0xD42C,0x03);
	spi_write_op(0xD42D,0x59);
	spi_write_op(0xD42E,0x03);
	spi_write_op(0xD42F,0x65);
	spi_write_op(0xD430,0x03);
	spi_write_op(0xD431,0x71);
	spi_write_op(0xD432,0x03);
	spi_write_op(0xD433,0xFF);

	spi_write_op(0xD500,0x00);
	spi_write_op(0xD501,0x00);
	spi_write_op(0xD502,0x00);
	spi_write_op(0xD503,0x11);
	spi_write_op(0xD504,0x00);
	spi_write_op(0xD505,0x2E);
	spi_write_op(0xD506,0x00);
	spi_write_op(0xD507,0x47);
	spi_write_op(0xD508,0x00);
	spi_write_op(0xD509,0x5C);
	spi_write_op(0xD50A,0x00);
	spi_write_op(0xD50B,0x7E);
	spi_write_op(0xD50C,0x00);
	spi_write_op(0xD50D,0x9C);
	spi_write_op(0xD50E,0x00);
	spi_write_op(0xD50F,0xCD);
	spi_write_op(0xD510,0x00);
	spi_write_op(0xD511,0xF6);
	spi_write_op(0xD512,0x01);
	spi_write_op(0xD513,0x36);
	spi_write_op(0xD514,0x01);
	spi_write_op(0xD515,0x6A);
	spi_write_op(0xD516,0x01);
	spi_write_op(0xD517,0xBD);
	spi_write_op(0xD518,0x01);
	spi_write_op(0xD519,0xFF);
	spi_write_op(0xD51A,0x02);
	spi_write_op(0xD51B,0x01);
	spi_write_op(0xD51C,0x02);
	spi_write_op(0xD51D,0x3B);
	spi_write_op(0xD51E,0x02);
	spi_write_op(0xD51F,0x75);
	spi_write_op(0xD520,0x02);
	spi_write_op(0xD521,0x99);
	spi_write_op(0xD522,0x02);
	spi_write_op(0xD523,0xC8);
	spi_write_op(0xD524,0x02);
	spi_write_op(0xD525,0xE8);
	spi_write_op(0xD526,0x03);
	spi_write_op(0xD527,0x11);
	spi_write_op(0xD528,0x03);
	spi_write_op(0xD529,0x2B);
	spi_write_op(0xD52A,0x03);
	spi_write_op(0xD52B,0x48);
	spi_write_op(0xD52C,0x03);
	spi_write_op(0xD52D,0x59);
	spi_write_op(0xD52E,0x03);
	spi_write_op(0xD52F,0x65);
	spi_write_op(0xD530,0x03);
	spi_write_op(0xD531,0x71);
	spi_write_op(0xD532,0x03);
	spi_write_op(0xD533,0xFF);

	spi_write_op(0xD600,0x00);
	spi_write_op(0xD601,0x00);
	spi_write_op(0xD602,0x00);
	spi_write_op(0xD603,0x11);
	spi_write_op(0xD604,0x00);
	spi_write_op(0xD605,0x2E);
	spi_write_op(0xD606,0x00);
	spi_write_op(0xD607,0x47);
	spi_write_op(0xD608,0x00);
	spi_write_op(0xD609,0x5C);
	spi_write_op(0xD60A,0x00);
	spi_write_op(0xD60B,0x7E);
	spi_write_op(0xD60C,0x00);
	spi_write_op(0xD60D,0x9C);
	spi_write_op(0xD60E,0x00);
	spi_write_op(0xD60F,0xCD);
	spi_write_op(0xD610,0x00);
	spi_write_op(0xD611,0xF6);
	spi_write_op(0xD612,0x01);
	spi_write_op(0xD613,0x36);
	spi_write_op(0xD614,0x01);
	spi_write_op(0xD615,0x6A);
	spi_write_op(0xD616,0x01);
	spi_write_op(0xD617,0xBD);
	spi_write_op(0xD618,0x01);
	spi_write_op(0xD619,0xFF);
	spi_write_op(0xD61A,0x02);
	spi_write_op(0xD61B,0x01);
	spi_write_op(0xD61C,0x02);
	spi_write_op(0xD61D,0x3B);
	spi_write_op(0xD61E,0x02);
	spi_write_op(0xD61F,0x75);
	spi_write_op(0xD620,0x02);
	spi_write_op(0xD621,0x99);
	spi_write_op(0xD622,0x02);
	spi_write_op(0xD623,0xC8);
	spi_write_op(0xD624,0x02);
	spi_write_op(0xD625,0xE8);
	spi_write_op(0xD626,0x03);
	spi_write_op(0xD627,0x11);
	spi_write_op(0xD628,0x03);
	spi_write_op(0xD629,0x2B);
	spi_write_op(0xD62A,0x03);
	spi_write_op(0xD62B,0x48);
	spi_write_op(0xD62C,0x03);
	spi_write_op(0xD62D,0x59);
	spi_write_op(0xD62E,0x03);
	spi_write_op(0xD62F,0x65);
	spi_write_op(0xD630,0x03);
	spi_write_op(0xD631,0x71);
	spi_write_op(0xD632,0x03);
	spi_write_op(0xD633,0xFF);

	udelay(10);

	spi_write_op(0xF000,0x55);
	spi_write_op(0xF001,0xAA);
	spi_write_op(0xF002,0x52);
	spi_write_op(0xF003,0x08);
	spi_write_op(0xF004,0x03);

	spi_write_op(0xB000,0x03);
	spi_write_op(0xB001,0x15);
	spi_write_op(0xB002,0xFA);
	spi_write_op(0xB003,0x00);
	spi_write_op(0xB004,0x00);
	spi_write_op(0xB005,0x00);
	spi_write_op(0xB006,0x00);

	spi_write_op(0xB200,0xFB);
	spi_write_op(0xB201,0xFC);
	spi_write_op(0xB202,0xFD);
	spi_write_op(0xB203,0xFE);
	spi_write_op(0xB204,0xF0);
	spi_write_op(0xB205,0x10);
	spi_write_op(0xB206,0x00);
	spi_write_op(0xB207,0x83);
	spi_write_op(0xB208,0x04);

	spi_write_op(0xB300,0x5B);
	spi_write_op(0xB301,0x00);
	spi_write_op(0xB302,0xFB);
	spi_write_op(0xB303,0x21);
	spi_write_op(0xB304,0x23);
	spi_write_op(0xB305,0x0C);

	spi_write_op(0xB400,0x00);
	spi_write_op(0xB401,0x00);
	spi_write_op(0xB402,0x00);
	spi_write_op(0xB403,0x00);
	spi_write_op(0xB404,0x00);
	spi_write_op(0xB405,0x00);
	spi_write_op(0xB406,0x00);
	spi_write_op(0xB407,0x00);
	spi_write_op(0xB408,0x00);

	spi_write_op(0xB500,0x00);
	spi_write_op(0xB501,0x00);
	spi_write_op(0xB502,0x00);
	spi_write_op(0xB503,0x00);
	spi_write_op(0xB504,0x00);
	spi_write_op(0xB505,0x00);
	spi_write_op(0xB506,0x00);
	spi_write_op(0xB507,0x00);
	spi_write_op(0xB508,0x00);
	spi_write_op(0xB509,0x00);
	spi_write_op(0xB50A,0x44);

	spi_write_op(0xB600,0x00);
	spi_write_op(0xB601,0x00);
	spi_write_op(0xB602,0x00);
	spi_write_op(0xB603,0x00);
	spi_write_op(0xB604,0x00);
	spi_write_op(0xB605,0x00);
	spi_write_op(0xB606,0x00);

	spi_write_op(0xB700,0x00);
	spi_write_op(0xB701,0x00);
	spi_write_op(0xB702,0x20);
	spi_write_op(0xB703,0x20);
	spi_write_op(0xB704,0x20);
	spi_write_op(0xB705,0x20);
	spi_write_op(0xB706,0x00);
	spi_write_op(0xB707,0x00);

	spi_write_op(0xB800,0x00);
	spi_write_op(0xB801,0x00);
	spi_write_op(0xB802,0x00);

	spi_write_op(0xB900,0x82);

	spi_write_op(0xBA00,0x45);
	spi_write_op(0xBA01,0x8A);
	spi_write_op(0xBA02,0x04);
	spi_write_op(0xBA03,0x5F);
	spi_write_op(0xBA04,0xFF);
	spi_write_op(0xBA05,0xFF);
	spi_write_op(0xBA06,0xFF);
	spi_write_op(0xBA07,0xFF);
	spi_write_op(0xBA08,0xFF);
	spi_write_op(0xBA09,0xFF);
	spi_write_op(0xBA0A,0xFF);
	spi_write_op(0xBA0B,0xFF);
	spi_write_op(0xBA0C,0xF5);
	spi_write_op(0xBA0D,0x41);
	spi_write_op(0xBA0E,0xB9);
	spi_write_op(0xBA0F,0x54);

	spi_write_op(0xBB00,0x54);
	spi_write_op(0xBB01,0xB9);
	spi_write_op(0xBB02,0x15);
	spi_write_op(0xBB03,0x4F);
	spi_write_op(0xBB04,0xFF);
	spi_write_op(0xBB05,0xFF);
	spi_write_op(0xBB06,0xFF);
	spi_write_op(0xBB07,0xFF);
	spi_write_op(0xBB08,0xFF);
	spi_write_op(0xBB09,0xFF);
	spi_write_op(0xBB0A,0xFF);
	spi_write_op(0xBB0B,0xFF);
	spi_write_op(0xBB0C,0xF4);
	spi_write_op(0xBB0D,0x50);
	spi_write_op(0xBB0E,0x8A);
	spi_write_op(0xBB0F,0x45);

	spi_write_op(0xBC00,0xC7);
	spi_write_op(0xBC01,0xFF);
	spi_write_op(0xBC02,0xFF);
	spi_write_op(0xBC03,0xE3);

	spi_write_op(0xBD00,0xC7);
	spi_write_op(0xBD01,0xFF);
	spi_write_op(0xBD02,0xFF);
	spi_write_op(0xBD03,0xE3);

	spi_write_op(0xC000,0x00);
	spi_write_op(0xC001,0x01);
	spi_write_op(0xC002,0xFA);
	spi_write_op(0xC003,0x00);

	 udelay(50);

	spi_write_op(0xF000,0x55);
	spi_write_op(0xF001,0xAA);
	spi_write_op(0xF002,0x52);
	spi_write_op(0xF003,0x08);
	spi_write_op(0xF004,0x00);

	spi_write_op(0xB000,0x00);
	spi_write_op(0xB001,0x10);

	spi_write_op(0xB500,0x50);

	spi_write_op(0xBA00,0x01);

	spi_write_op(0xBC00,0x00);
	spi_write_op(0xBC01,0x00);
	spi_write_op(0xBC02,0x00);

	spi_write_cmd(0x1100);

	udelay(120);

	spi_write_cmd(0x2900);

	udelay(120);
#elif (defined PANEL_RM68172_5INCH)
	spi_write_op(0xF000,0x55);
	spi_write_op(0xF001,0xAA);
	spi_write_op(0xF002,0x52);
	spi_write_op(0xF003,0x08);
	spi_write_op(0xF004,0x02);

	spi_write_op(0xF600,0x60);
	spi_write_op(0xF601,0x40);

	spi_write_op(0xFE00,0x01);
	spi_write_op(0xFE01,0x80);
	spi_write_op(0xFE02,0x09);
	spi_write_op(0xFE03,0x09);

	spi_write_op(0xF000,0x55);
	spi_write_op(0xF001,0xAA);
	spi_write_op(0xF002,0x52);
	spi_write_op(0xF003,0x08);
	spi_write_op(0xF004,0x01);

	spi_write_op(0xB000,0x0D);

	spi_write_op(0xB100,0x0D);

	spi_write_op(0xB500,0x08);

	spi_write_op(0xB600,0x54);

	spi_write_op(0xB700,0x44);

	spi_write_op(0xB800,0x24);

	spi_write_op(0xB900,0x34);

	spi_write_op(0xBA00,0x14);

	spi_write_op(0xBC00,0x00);
	spi_write_op(0xBC01,0x98);//78
	spi_write_op(0xBC02,0x00);

	spi_write_op(0xBD00,0x00);
	spi_write_op(0xBD01,0x98);//78
	spi_write_op(0xBD02,0x00);

	spi_write_op(0xBE00,0x00);
	spi_write_op(0xBE01,0x16);

	spi_write_op(0xD100,0x00);
	spi_write_op(0xD101,0x00);
	spi_write_op(0xD102,0x00);
	spi_write_op(0xD103,0x18);
	spi_write_op(0xD104,0x00);
	spi_write_op(0xD105,0x3A);
	spi_write_op(0xD106,0x00);
	spi_write_op(0xD107,0x58);
	spi_write_op(0xD108,0x00);
	spi_write_op(0xD109,0x70);
	spi_write_op(0xD10A,0x00);
	spi_write_op(0xD10B,0x9B);
	spi_write_op(0xD10C,0x00);
	spi_write_op(0xD10D,0xBD);
	spi_write_op(0xD10E,0x00);
	spi_write_op(0xD10F,0xF2);
	spi_write_op(0xD110,0x01);
	spi_write_op(0xD111,0x1B);
	spi_write_op(0xD112,0x01);
	spi_write_op(0xD113,0x58);
	spi_write_op(0xD114,0x01);
	spi_write_op(0xD115,0x87);
	spi_write_op(0xD116,0x01);
	spi_write_op(0xD117,0xCC);
	spi_write_op(0xD118,0x02);
	spi_write_op(0xD119,0x01);
	spi_write_op(0xD11A,0x02);
	spi_write_op(0xD11B,0x02);
	spi_write_op(0xD11C,0x02);
	spi_write_op(0xD11D,0x35);
	spi_write_op(0xD11E,0x02);
	spi_write_op(0xD11F,0x70);
	spi_write_op(0xD120,0x02);
	spi_write_op(0xD121,0x96);
	spi_write_op(0xD122,0x02);
	spi_write_op(0xD123,0xC7);
	spi_write_op(0xD124,0x02);
	spi_write_op(0xD125,0xE8);
	spi_write_op(0xD126,0x03);
	spi_write_op(0xD127,0x14);
	spi_write_op(0xD128,0x03);
	spi_write_op(0xD129,0x2D);
	spi_write_op(0xD12A,0x03);
	spi_write_op(0xD12B,0x4A);
	spi_write_op(0xD12C,0x03);
	spi_write_op(0xD12D,0x5C);
	spi_write_op(0xD12E,0x03);
	spi_write_op(0xD12F,0x6A);
	spi_write_op(0xD130,0x03);
	spi_write_op(0xD131,0x6B);
	spi_write_op(0xD132,0x03);
	spi_write_op(0xD133,0xFF);

	spi_write_op(0xD200,0x00);
	spi_write_op(0xD201,0x00);
	spi_write_op(0xD202,0x00);
	spi_write_op(0xD203,0x18);
	spi_write_op(0xD204,0x00);
	spi_write_op(0xD205,0x3A);
	spi_write_op(0xD206,0x00);
	spi_write_op(0xD207,0x58);
	spi_write_op(0xD208,0x00);
	spi_write_op(0xD209,0x70);
	spi_write_op(0xD20A,0x00);
	spi_write_op(0xD20B,0x9B);
	spi_write_op(0xD20C,0x00);
	spi_write_op(0xD20D,0xBD);
	spi_write_op(0xD20E,0x00);
	spi_write_op(0xD20F,0xF2);
	spi_write_op(0xD210,0x01);
	spi_write_op(0xD211,0x1B);
	spi_write_op(0xD212,0x01);
	spi_write_op(0xD213,0x58);
	spi_write_op(0xD214,0x01);
	spi_write_op(0xD215,0x87);
	spi_write_op(0xD216,0x01);
	spi_write_op(0xD217,0xCC);
	spi_write_op(0xD218,0x02);
	spi_write_op(0xD219,0x01);
	spi_write_op(0xD21A,0x02);
	spi_write_op(0xD21B,0x02);
	spi_write_op(0xD21C,0x02);
	spi_write_op(0xD21D,0x35);
	spi_write_op(0xD21E,0x02);
	spi_write_op(0xD21F,0x70);
	spi_write_op(0xD220,0x02);
	spi_write_op(0xD221,0x96);
	spi_write_op(0xD222,0x02);
	spi_write_op(0xD223,0xC7);
	spi_write_op(0xD224,0x02);
	spi_write_op(0xD225,0xE8);
	spi_write_op(0xD226,0x03);
	spi_write_op(0xD227,0x14);
	spi_write_op(0xD228,0x03);
	spi_write_op(0xD229,0x2D);
	spi_write_op(0xD22A,0x03);
	spi_write_op(0xD22B,0x4A);
	spi_write_op(0xD22C,0x03);
	spi_write_op(0xD22D,0x5C);
	spi_write_op(0xD22E,0x03);
	spi_write_op(0xD22F,0x6A);
	spi_write_op(0xD230,0x03);
	spi_write_op(0xD231,0x6B);
	spi_write_op(0xD232,0x03);
	spi_write_op(0xD233,0xFF);

	spi_write_op(0xD300,0x00);
	spi_write_op(0xD301,0x00);
	spi_write_op(0xD302,0x00);
	spi_write_op(0xD303,0x18);
	spi_write_op(0xD304,0x00);
	spi_write_op(0xD305,0x3A);
	spi_write_op(0xD306,0x00);
	spi_write_op(0xD307,0x58);
	spi_write_op(0xD308,0x00);
	spi_write_op(0xD309,0x70);
	spi_write_op(0xD30A,0x00);
	spi_write_op(0xD30B,0x9B);
	spi_write_op(0xD30C,0x00);
	spi_write_op(0xD30D,0xBD);
	spi_write_op(0xD30E,0x00);
	spi_write_op(0xD30F,0xF2);
	spi_write_op(0xD310,0x01);
	spi_write_op(0xD311,0x1B);
	spi_write_op(0xD312,0x01);
	spi_write_op(0xD313,0x58);
	spi_write_op(0xD314,0x01);
	spi_write_op(0xD315,0x87);
	spi_write_op(0xD316,0x01);
	spi_write_op(0xD317,0xCC);
	spi_write_op(0xD318,0x02);
	spi_write_op(0xD319,0x01);
	spi_write_op(0xD31A,0x02);
	spi_write_op(0xD31B,0x02);
	spi_write_op(0xD31C,0x02);
	spi_write_op(0xD31D,0x35);
	spi_write_op(0xD31E,0x02);
	spi_write_op(0xD31F,0x70);
	spi_write_op(0xD320,0x02);
	spi_write_op(0xD321,0x96);
	spi_write_op(0xD322,0x02);
	spi_write_op(0xD323,0xC7);
	spi_write_op(0xD324,0x02);
	spi_write_op(0xD325,0xE8);
	spi_write_op(0xD326,0x03);
	spi_write_op(0xD327,0x14);
	spi_write_op(0xD328,0x03);
	spi_write_op(0xD329,0x2D);
	spi_write_op(0xD32A,0x03);
	spi_write_op(0xD32B,0x4A);
	spi_write_op(0xD32C,0x03);
	spi_write_op(0xD32D,0x5C);
	spi_write_op(0xD32E,0x03);
	spi_write_op(0xD32F,0x6A);
	spi_write_op(0xD330,0x03);
	spi_write_op(0xD331,0x6B);
	spi_write_op(0xD332,0x03);
	spi_write_op(0xD333,0xFF);

	spi_write_op(0xD400,0x00);
	spi_write_op(0xD401,0x00);
	spi_write_op(0xD402,0x00);
	spi_write_op(0xD403,0x18);
	spi_write_op(0xD404,0x00);
	spi_write_op(0xD405,0x3A);
	spi_write_op(0xD406,0x00);
	spi_write_op(0xD407,0x58);
	spi_write_op(0xD408,0x00);
	spi_write_op(0xD409,0x70);
	spi_write_op(0xD40A,0x00);
	spi_write_op(0xD40B,0x9B);
	spi_write_op(0xD40C,0x00);
	spi_write_op(0xD40D,0xBD);
	spi_write_op(0xD40E,0x00);
	spi_write_op(0xD40F,0xF2);
	spi_write_op(0xD410,0x01);
	spi_write_op(0xD411,0x1B);
	spi_write_op(0xD412,0x01);
	spi_write_op(0xD413,0x58);
	spi_write_op(0xD414,0x01);
	spi_write_op(0xD415,0x87);
	spi_write_op(0xD416,0x01);
	spi_write_op(0xD417,0xCC);
	spi_write_op(0xD418,0x02);
	spi_write_op(0xD419,0x01);
	spi_write_op(0xD41A,0x02);
	spi_write_op(0xD41B,0x02);
	spi_write_op(0xD41C,0x02);
	spi_write_op(0xD41D,0x35);
	spi_write_op(0xD41E,0x02);
	spi_write_op(0xD41F,0x70);
	spi_write_op(0xD420,0x02);
	spi_write_op(0xD421,0x96);
	spi_write_op(0xD422,0x02);
	spi_write_op(0xD423,0xC7);
	spi_write_op(0xD424,0x02);
	spi_write_op(0xD425,0xE8);
	spi_write_op(0xD426,0x03);
	spi_write_op(0xD427,0x14);
	spi_write_op(0xD428,0x03);
	spi_write_op(0xD429,0x2D);
	spi_write_op(0xD42A,0x03);
	spi_write_op(0xD42B,0x4A);
	spi_write_op(0xD42C,0x03);
	spi_write_op(0xD42D,0x5C);
	spi_write_op(0xD42E,0x03);
	spi_write_op(0xD42F,0x6A);
	spi_write_op(0xD430,0x03);
	spi_write_op(0xD431,0x6B);
	spi_write_op(0xD432,0x03);
	spi_write_op(0xD433,0xFF);

	spi_write_op(0xD500,0x00);
	spi_write_op(0xD501,0x00);
	spi_write_op(0xD502,0x00);
	spi_write_op(0xD503,0x18);
	spi_write_op(0xD504,0x00);
	spi_write_op(0xD505,0x3A);
	spi_write_op(0xD506,0x00);
	spi_write_op(0xD507,0x58);
	spi_write_op(0xD508,0x00);
	spi_write_op(0xD509,0x70);
	spi_write_op(0xD50A,0x00);
	spi_write_op(0xD50B,0x9B);
	spi_write_op(0xD50C,0x00);
	spi_write_op(0xD50D,0xBD);
	spi_write_op(0xD50E,0x00);
	spi_write_op(0xD50F,0xF2);
	spi_write_op(0xD510,0x01);
	spi_write_op(0xD511,0x1B);
	spi_write_op(0xD512,0x01);
	spi_write_op(0xD513,0x58);
	spi_write_op(0xD514,0x01);
	spi_write_op(0xD515,0x87);
	spi_write_op(0xD516,0x01);
	spi_write_op(0xD517,0xCC);
	spi_write_op(0xD518,0x02);
	spi_write_op(0xD519,0x01);
	spi_write_op(0xD51A,0x02);
	spi_write_op(0xD51B,0x02);
	spi_write_op(0xD51C,0x02);
	spi_write_op(0xD51D,0x35);
	spi_write_op(0xD51E,0x02);
	spi_write_op(0xD51F,0x70);
	spi_write_op(0xD520,0x02);
	spi_write_op(0xD521,0x96);
	spi_write_op(0xD522,0x02);
	spi_write_op(0xD523,0xC7);
	spi_write_op(0xD524,0x02);
	spi_write_op(0xD525,0xE8);
	spi_write_op(0xD526,0x03);
	spi_write_op(0xD527,0x14);
	spi_write_op(0xD528,0x03);
	spi_write_op(0xD529,0x2D);
	spi_write_op(0xD52A,0x03);
	spi_write_op(0xD52B,0x4A);
	spi_write_op(0xD52C,0x03);
	spi_write_op(0xD52D,0x5C);
	spi_write_op(0xD52E,0x03);
	spi_write_op(0xD52F,0x6A);
	spi_write_op(0xD530,0x03);
	spi_write_op(0xD531,0x6B);
	spi_write_op(0xD532,0x03);
	spi_write_op(0xD533,0xFF);

	spi_write_op(0xD600,0x00);
	spi_write_op(0xD601,0x00);
	spi_write_op(0xD602,0x00);
	spi_write_op(0xD603,0x18);
	spi_write_op(0xD604,0x00);
	spi_write_op(0xD605,0x3A);
	spi_write_op(0xD606,0x00);
	spi_write_op(0xD607,0x58);
	spi_write_op(0xD608,0x00);
	spi_write_op(0xD609,0x70);
	spi_write_op(0xD60A,0x00);
	spi_write_op(0xD60B,0x9B);
	spi_write_op(0xD60C,0x00);
	spi_write_op(0xD60D,0xBD);
	spi_write_op(0xD60E,0x00);
	spi_write_op(0xD60F,0xF2);
	spi_write_op(0xD610,0x01);
	spi_write_op(0xD611,0x1B);
	spi_write_op(0xD612,0x01);
	spi_write_op(0xD613,0x58);
	spi_write_op(0xD614,0x01);
	spi_write_op(0xD615,0x87);
	spi_write_op(0xD616,0x01);
	spi_write_op(0xD617,0xCC);
	spi_write_op(0xD618,0x02);
	spi_write_op(0xD619,0x01);
	spi_write_op(0xD61A,0x02);
	spi_write_op(0xD61B,0x02);
	spi_write_op(0xD61C,0x02);
	spi_write_op(0xD61D,0x35);
	spi_write_op(0xD61E,0x02);
	spi_write_op(0xD61F,0x70);
	spi_write_op(0xD620,0x02);
	spi_write_op(0xD621,0x96);
	spi_write_op(0xD622,0x02);
	spi_write_op(0xD623,0xC7);
	spi_write_op(0xD624,0x02);
	spi_write_op(0xD625,0xE8);
	spi_write_op(0xD626,0x03);
	spi_write_op(0xD627,0x14);
	spi_write_op(0xD628,0x03);
	spi_write_op(0xD629,0x2D);
	spi_write_op(0xD62A,0x03);
	spi_write_op(0xD62B,0x4A);
	spi_write_op(0xD62C,0x03);
	spi_write_op(0xD62D,0x5C);
	spi_write_op(0xD62E,0x03);
	spi_write_op(0xD62F,0x6A);
	spi_write_op(0xD630,0x03);
	spi_write_op(0xD631,0x6B);
	spi_write_op(0xD632,0x03);
	spi_write_op(0xD633,0xFF);

	spi_write_op(0xF000,0x55);
	spi_write_op(0xF001,0xAA);
	spi_write_op(0xF002,0x52);
	spi_write_op(0xF003,0x08);
	spi_write_op(0xF004,0x03);

	spi_write_op(0xB000,0x05);
	spi_write_op(0xB001,0x17);
	spi_write_op(0xB002,0xF9);
	spi_write_op(0xB003,0x53);
	spi_write_op(0xB004,0x00);
	spi_write_op(0xB005,0x00);
	spi_write_op(0xB006,0x30);

	spi_write_op(0xB100,0x05);
	spi_write_op(0xB101,0x17);
	spi_write_op(0xB102,0xFB);
	spi_write_op(0xB103,0x55);
	spi_write_op(0xB104,0x00);
	spi_write_op(0xB105,0x00);
	spi_write_op(0xB106,0x30);

	spi_write_op(0xB200,0xFC);
	spi_write_op(0xB201,0xFD);
	spi_write_op(0xB202,0xFE);
	spi_write_op(0xB203,0xFF);
	spi_write_op(0xB204,0xF0);
	spi_write_op(0xB205,0xED);
	spi_write_op(0xB206,0x28);
	spi_write_op(0xB207,0xC4);
	spi_write_op(0xB208,0x08);

	spi_write_op(0xB300,0x5B);
	spi_write_op(0xB301,0x00);
	spi_write_op(0xB302,0xFC);
	spi_write_op(0xB303,0x5A);
	spi_write_op(0xB304,0x5A);
	spi_write_op(0xB305,0x0C);

	spi_write_op(0xB400,0x00);
	spi_write_op(0xB401,0x01);
	spi_write_op(0xB402,0x02);
	spi_write_op(0xB403,0x03);
	spi_write_op(0xB404,0x00);
	spi_write_op(0xB405,0x40);
	spi_write_op(0xB406,0x04);
	spi_write_op(0xB407,0x08);
	spi_write_op(0xB408,0xED);
	spi_write_op(0xB409,0x28);
	spi_write_op(0xB40A,0x00);

	spi_write_op(0xB500,0x00);
	spi_write_op(0xB501,0x00);
	spi_write_op(0xB502,0x00);
	spi_write_op(0xB503,0x80);
	spi_write_op(0xB504,0x60);
	spi_write_op(0xB505,0x5E);
	spi_write_op(0xB506,0x50);
	spi_write_op(0xB507,0x50);
	spi_write_op(0xB508,0x33);
	spi_write_op(0xB509,0x33);
	spi_write_op(0xB50A,0x55);

	spi_write_op(0xB600,0xBC);
	spi_write_op(0xB601,0x00);
	spi_write_op(0xB602,0x00);
	spi_write_op(0xB603,0x00);
	spi_write_op(0xB604,0x96);
	spi_write_op(0xB605,0x81);
	spi_write_op(0xB606,0x00);

	spi_write_op(0xB700,0x00);
	spi_write_op(0xB701,0x00);
	spi_write_op(0xB702,0x00);
	spi_write_op(0xB703,0x00);
	spi_write_op(0xB704,0x00);
	spi_write_op(0xB705,0x00);
	spi_write_op(0xB706,0x00);
	spi_write_op(0xB707,0x00);

	spi_write_op(0xB800,0x11);
	spi_write_op(0xB801,0x18);
	spi_write_op(0xB802,0x00);

	spi_write_op(0xB900,0x90);

	spi_write_op(0xBA00,0x44);
	spi_write_op(0xBA01,0x44);
	spi_write_op(0xBA02,0x08);
	spi_write_op(0xBA03,0xAC);
	spi_write_op(0xBA04,0xE2);
	spi_write_op(0xBA05,0x64);
	spi_write_op(0xBA06,0x44);
	spi_write_op(0xBA07,0x44);
	spi_write_op(0xBA08,0x44);
	spi_write_op(0xBA09,0x44);
	spi_write_op(0xBA0A,0x47);
	spi_write_op(0xBA0B,0x3F);
	spi_write_op(0xBA0C,0xDB);
	spi_write_op(0xBA0D,0x91);
	spi_write_op(0xBA0E,0x54);
	spi_write_op(0xBA0F,0x44);

	spi_write_op(0xBB00,0x44);
	spi_write_op(0xBB01,0x43);
	spi_write_op(0xBB02,0x79);
	spi_write_op(0xBB03,0xFD);
	spi_write_op(0xBB04,0xB5);
	spi_write_op(0xBB05,0x14);
	spi_write_op(0xBB06,0x44);
	spi_write_op(0xBB07,0x44);
	spi_write_op(0xBB08,0x44);
	spi_write_op(0xBB09,0x44);
	spi_write_op(0xBB0A,0x40);
	spi_write_op(0xBB0B,0x4A);
	spi_write_op(0xBB0C,0xCE);
	spi_write_op(0xBB0D,0x86);
	spi_write_op(0xBB0E,0x24);
	spi_write_op(0xBB0F,0x44);

	spi_write_op(0xBC00,0xE0);
	spi_write_op(0xBC01,0x1F);
	spi_write_op(0xBC02,0xF8);
	spi_write_op(0xBC03,0x07);

	spi_write_op(0xBD00,0xE0);
	spi_write_op(0xBD01,0x1F);
	spi_write_op(0xBD02,0xF8);
	spi_write_op(0xBD03,0x07);

	spi_write_op(0xF000,0x55);
	spi_write_op(0xF001,0xAA);
	spi_write_op(0xF002,0x52);
	spi_write_op(0xF003,0x08);
	spi_write_op(0xF004,0x00);

	spi_write_op(0xB000,0x00);
	spi_write_op(0xB001,0x10);

	spi_write_op(0xB500,0x6B);

	spi_write_op(0xB800,0x00);
	spi_write_op(0xB801,0x03);
	spi_write_op(0xB802,0x03);
	spi_write_op(0xB803,0x03);

	spi_write_op(0xBC00,0x00);

	spi_write_op(0x3500,0x00);

	spi_write_cmd(0x1100);

	udelay(120);

	spi_write_cmd(0x2900);

	udelay(120);
#endif

	return 0;
}

static void vpl_voc_start(void)
{
	/* Enable VOC Operation */
	voc->dwCtrl |= (1 << 2);
}

static void video_hw_init(void)
{
	u32 val;
	/* io pad mux done by loader */

	/* sysc voc clock enable */
	val = readl(VPL_SYSC_MMR_BASE + SYSC_MMR_CLKEN_CTRL1);
	val |= VOC_CLK_EN_BIT_MASK;
	writel(val, VPL_SYSC_MMR_BASE + SYSC_MMR_CLKEN_CTRL1);

	/* soft reset */
	writel(readl(VPL_SYSC_MMR_BASE) & (~VOC_RST_N_BIT_MASK), SYSC_RST_CTRL);
	writel(readl(VPL_SYSC_MMR_BASE) | VOC_RST_N_BIT_MASK, SYSC_RST_CTRL);

	debug("voc version %x\n", voc->dwVersion);
	/* voc initialize */
	/* Setup Default SBC */
		/* Skip, use default value */

	/* Setup Default Color Space Conversion Matrix */
		/* Skip use default value*/

	/* Set Input formate*/
#if (defined PANEL_NT39016)
    voc->dwCtrl = (0x1 << 29) | (0x78 << 16) | (1 << 10) | (2 << 11) | (1 << 15);
#elif (defined PANEL_ILI9341V)
    voc->dwCtrl = (0x1 << 29) | (0x78 << 16) | (1 << 10) | (2 << 11) | (1 << 15) | (4 << 24);
#elif (defined PANEL_ILI9342C) || (defined PANEL_ST7789V)
    voc->dwCtrl = (0x1 << 29) | (0x78 << 16) | (1 << 10) | (2 << 11) | (1 << 15);
#elif (defined PANEL_ST7789) || (defined PANEL_ST7793)
    voc->dwCtrl = (0x1 << 29) | (0x78 << 16) | (1 << 10) | (2 << 11) | (1 << 14);
#elif (defined PANEL_RM68172_4INCH) || (defined PANEL_RM68172_5INCH)
    voc->dwCtrl = (0x1 << 29) | (0x78 << 16) | (1 << 10) | (0x04 << 24);
#endif

	/* Setup Input buffer stride */
	writel((modes.hactive)<<16, &voc->dwStrideSetting);

	/* Setup Output Signal Timings */
	update_display_mode();
#if 0
	debug("dwVersion          %08x\n", voc->dwVersion);
	debug("dwCtrl             %08x\n", voc->dwCtrl);
	debug("dwStat             %08x\n", voc->dwStat);
	debug("dwYBuff0Addr       %08x\n", voc->dwYBuff0Addr);
	debug("dwCbBuff0Addr      %08x\n", voc->dwCbBuff0Addr);
	debug("dwCrBuff0Addr      %08x\n", voc->dwCrBuff0Addr);
	debug("dwYBuff1Addr       %08x\n", voc->dwYBuff1Addr);
	debug("dwCbBuff1Addr      %08x\n", voc->dwCbBuff1Addr);
	debug("dwCrBuff1Addr      %08x\n", voc->dwCrBuff1Addr);
	debug("dwInSize           %08x\n", voc->dwInSize);
	debug("dwOutSize          %08x\n", voc->dwOutSize);
	debug("dwOutSizeCtrl      %08x\n", voc->dwOutSizeCtrl);
	debug("dwHSyncCtrl        %08x\n", voc->dwHSyncCtrl);
	debug("dwVSyncCtrl        %08x\n", voc->dwVSyncCtrl);
	debug("dwSBCCtrl          %08x\n", voc->dwSBCCtrl);
	debug("dwYCbCr2RGBCoeff0  %08x\n", voc->dwYCbCr2RGBCoeff0);
	debug("dwYCbCr2RGBCoeff1  %08x\n", voc->dwYCbCr2RGBCoeff1);
	debug("dwYCbCr2RGBCoeff2  %08x\n", voc->dwYCbCr2RGBCoeff2);
	debug("dwStrideSetting    %08x\n", voc->dwStrideSetting);
#endif
}

#ifdef CONFIG_SPI_FLASH
static struct spi_flash *sf;

extern struct spi_flash *get_spi_flash(void);
static int raw_sf_read(u32 bmp_load_addr, int offset, size_t read_size)
{
	if (!sf) {
		sf = get_spi_flash();
		if (!sf)
			return -ENODEV;
	}
	debug("bmp_load_addr %x %x %d\n", bmp_load_addr, offset, read_size);
	return spi_flash_read(sf, offset, read_size, (void *)bmp_load_addr);
}
#else
static int raw_sf_read(u32 bmp_load_addr, int offset, size_t read_size)
{
	debug("%s: sf support not available\n", __func__);
	return -ENOSYS;
}
#endif

#ifdef CONFIG_CMD_NAND
static int raw_nand_read(u32 bmp_load_addr, int offset, size_t read_size)
{
	return nand_read_skip_bad(&nand_info[nand_curr_device], offset,
				  &read_size, (u_char *)bmp_load_addr);
}
#else
static int raw_nand_read(u32 bmp_load_addr, int offset, size_t read_size)
{
	debug("%s: nand support not available\n", __func__);
	return -ENOSYS;
}
#endif

static int splash_storage_read(struct splash_location *location,
			       u32 bmp_load_addr, size_t read_size)
{
	u32 offset;

	if (!location)
		return -EINVAL;

	offset = location->offset;
	switch (location->storage) {
	case SPLASH_STORAGE_NAND:
		return raw_nand_read(bmp_load_addr, offset, read_size);
	case SPLASH_STORAGE_SF:
		return raw_sf_read(bmp_load_addr, offset, read_size);
	default:
		debug("Unknown splash location\n");
	}

	return -EINVAL;
}

static struct splash_location vpl_voc_splash_locations[] = {
	{
		.name = "sf",
		.storage = SPLASH_STORAGE_SF,
		.offset = 0x00FC0000,
	},
    {
        .name = "n2k",
        .storage = SPLASH_STORAGE_NAND,
        .offset = 0x07F40000,
    },
};

int splash_screen_prepare(u32 addr, size_t size)
{
#if (PLATFORM_SPIFLASH_NUM != 0)
    return splash_storage_read(&vpl_voc_splash_locations[0], addr, size);
#elif (PLATFORM_NANDFLASH_NUM != 0 || PLATFORM_SPI_NANDFLASH_NUM != 0)
	return splash_storage_read(&vpl_voc_splash_locations[1], addr, size);
#endif
}

int drv_video_init(void)
{
	u32 addr = 0;
	char *s;
	u32 Yframe_size = modes.hactive * modes.vactive;
	u32 Cbframe_size = (Yframe_size >> 2);

	u32 fbmem_size = (modes.hactive * modes.vactive)*3 >> 1;

	video_hw_init();

	s = getenv("splashimage");
	if (s != NULL) {
		addr = simple_strtoul(s, NULL, 16);
	} else {
#if 0
		addr = (u32)memalign(8, fbmem_size);
#else
		//addr = 0x10000000 - 0x40000;
		addr = gd->bd->bi_dram[0].size - ((fbmem_size + 4095) & (~4095));
#endif
	}

	if (addr == 0) {
		debug("Failed to alloc FB memory\n");
		return -1;
	}

	splash_screen_prepare(addr, fbmem_size);

	voc->dwYBuff1Addr = addr;
	voc->dwCbBuff1Addr = addr+Yframe_size;
	voc->dwCrBuff1Addr = addr+Yframe_size+Cbframe_size;

	lcd_init();

	vpl_voc_start();
	return 0;
}
