
#include <common.h>
#include <asm/io.h>
#include <asm/arch/platform.h>
#include <div64.h>
#include "vpl_voc.h"

static unsigned long gcd(unsigned long a, unsigned long b)
{
	unsigned long r;

	if (a < b)
		swap(a, b);

	if (!b)
		return a;
	while ((r = a % b) != 0) {
		a = b;
		b = r;
	}
	return b;
}

static void pllc_reduce_ratio(unsigned *nom, unsigned *den,
		unsigned nom_min, unsigned den_min)
{
	unsigned tmp;
	/* reduce the numbers to a simpler ratio */
	tmp = gcd(*nom, *den);
	*nom /= tmp;
	*den /= tmp;

	/* make sure nominator is large enough */
	if (*nom < nom_min) {
		tmp = DIV_ROUND_UP(nom_min, *nom);
		*nom *= tmp;
		*den *= tmp;
	}

	/* make sure the denominator is large enough */
	if (*den < den_min) {
		tmp = DIV_ROUND_UP(den_min, *den);
		*nom *= tmp;
		*den *= tmp;
	}
}

static void pllc_get_fb_ref_div(unsigned nom, unsigned den, unsigned post_div,
                                 unsigned fb_div_max, unsigned ref_div_max,
                                 unsigned *fb_div, unsigned *ref_div)
{
        /* limit reference * post divider to a maximum */
        ref_div_max = max(min(100 / post_div, ref_div_max), 1u);

        /* get matching reference and feedback divider */
        *ref_div = min(max(DIV_ROUND_CLOSEST(den, post_div), 1u), ref_div_max);
        *fb_div = DIV_ROUND_CLOSEST(nom * *ref_div * post_div, den);

        /* limit fb divider to its maximum */
        if (*fb_div > fb_div_max) {
                *ref_div = DIV_ROUND_CLOSEST(*ref_div * fb_div_max, *fb_div);
                *fb_div = fb_div_max;
        }
}

static int voc_compute_pll(u32 f_target, u32 *f_estim_p, u32 *fb_div_p, u32 *ref_div_p, u32 *post_div_p)
{
	u32 pllc_ref_req = 12000000;
	u32 pllc_min_ref_div = 1;
	u32 pllc_max_ref_div = 64;
	u32 pllc_min_fb_div = 8;
	u32 pllc_max_fb_div = 65536;
	u32 pllc_min_post_div = 1;
	u32 pllc_max_post_div = 32;
	u32 pllc_min_vco = 800000000;
	u32 pllc_max_vco = 1600000000;

	u32 target_clock = f_target;
	u32 fb_div_min, fb_div_max, fb_div;
	u32 post_div_min, post_div_max, post_div;
	u32 ref_div_min, ref_div_max, ref_div;
	u32 post_div_best, diff_best;
	u32 nom, den;
	u64 f_estim;

	/* determine allowed feedback divider range */
	fb_div_min = pllc_min_fb_div;
	fb_div_max = pllc_max_fb_div;

	/* determine allowed ref divider range */
	ref_div_min = pllc_min_ref_div;
	ref_div_max = pllc_max_ref_div;

	/* determine allowed post divider range */
	/* round to even number? */
	post_div_min = max(pllc_min_post_div, DIV_ROUND_UP(pllc_min_vco, target_clock));
	post_div_max = min(pllc_max_post_div, pllc_max_vco/target_clock);
	if (post_div_min > post_div_max) {
		debug("no available post_div\n");
		return -EINVAL;
	}

	/* represent the searched ratio as fractional number */
	nom = target_clock;
	den = pllc_ref_req;

	/* reduce the numbers to a simpler ratio */
	pllc_reduce_ratio(&nom, &den, fb_div_min, post_div_min);

	/* now search for a post divider */
	post_div_best = post_div_min;

	diff_best = ~0;

	for (post_div = post_div_min; post_div <= post_div_max; ++post_div) {
		unsigned diff;
		/* Skip odd post divider */
		if (post_div & 1)
			continue;

		pllc_get_fb_ref_div(nom, den, post_div, fb_div_max,
				ref_div_max, &fb_div, &ref_div);
		diff = abs(target_clock - (pllc_ref_req * fb_div) /
				(ref_div * post_div));

		if (diff < diff_best) {
			post_div_best = post_div;
			diff_best = diff;
		}
	}
	post_div = post_div_best;

	/* get the feedback and reference divider for the optimal value */
	pllc_get_fb_ref_div(nom, den, post_div, fb_div_max, ref_div_max, &fb_div, &ref_div);

	/* reduce the numbers to a simpler ratio once more */
	/* this also makes sure that the reference divider is large enough */
	pllc_reduce_ratio(&fb_div, &ref_div, fb_div_min, ref_div_min);

	/* avoid high jitter with small fractional dividers */
	f_estim = (u64)pllc_ref_req * fb_div;
	do_div(f_estim, (ref_div * post_div));

	*fb_div_p = fb_div;
	*ref_div_p = ref_div;
	*post_div_p = post_div;
	*f_estim_p = f_estim;

	return 0;
}

int voc_compute_pll_and_div(u32 pclk_target, u32 *f_estim, u32 *fb_div_p, u32 *ref_div_p, u32 *post_div_p, u32 *voc_div_p)
{
	u32 pllc_min_out = 25000000;
	u32 pllc_max_out = 1600000000;
	u32 voc_min_div = 1;
	u32 voc_max_div = 16;
	u32 voc_div_min, voc_div_max, voc_div;
	u32 fb_div, ref_div, post_div;
	u32 diff_best;
	int ret = -EINVAL;

	voc_div_min = max(voc_min_div, DIV_ROUND_UP(pllc_min_out, pclk_target * 4));
	voc_div_max = min(voc_max_div, pllc_max_out/(pclk_target*4));

	diff_best = ~0;

	for (voc_div = voc_div_min; voc_div <= voc_div_max; ++voc_div) {
		unsigned diff, pll_target, pll_out;
		pll_target = pclk_target * 4 * voc_div;
		ret = voc_compute_pll(pll_target, &pll_out, &fb_div, &ref_div, &post_div);
		if (ret)
			continue;

		diff = abs(pclk_target - pll_out / (4 * voc_div));
		if (diff < diff_best) {
			diff_best = diff;
			*fb_div_p = fb_div;
			*ref_div_p = ref_div;
			*post_div_p = post_div;
			*voc_div_p = voc_div;
			*f_estim = pll_out / (4 * voc_div);
		}
	}

	return ret;
}
